/*!
 *         @file    $Id:: test_HMC_Clover_SF_Leapfrog_Nf2.cpp #$
 *
 *         @brief   Testing the HMC leapfrog step with SF BC.
 *
 *         @author  Yusuke Taniguchi <tanigchi@het.ph.tsukuba.ac.jp> (taniguchi)
 *                  $LastChangedBy: sueda $
 *
 *         @date    $LastChangedDate:: 2013-07-19 14:15:23 #$
 *
 *         @version $LastChangedRevision: 936 $
 */

#include "parameterManager_YAML.h"
#include "parameters_factory.h"

#include "bridgeIO.h"
using Bridge::vout;

#include "gaugeConfig.h"

#include "staples_SF.h"
#include "action_G_Plaq_SF.h"
#include "action_G_Rectangle_SF.h"

#include "fopr_Clover_SF.h"
#include "force_F_Clover_SF.h"

#include "randomNumbers_Mseries.h"
#include "randomNumbers_MT19937.h"

#include "director_Smear.h"
#include "fopr_Smeared.h"
#include "force_F_Smeared.h"

#include "action_F_Standard_SF.h"

#include "forceSmear.h"
#include "projection.h"
#include "smear.h"
#include "solver.h"

#include "hmc_General.h"
#include "builder_Integrator.h"
#include "integrator.h"

#ifdef USE_TEST
#include "test.h"
#endif

#ifdef USE_TESTMANAGER_AUTOREGISTER
#include "testManager.h"
#endif

//====================================================================
//! Test of HMC update for clover fermions with SF.

/*!
    Multi-step leapfrog integrator for SF formalism.
    <ul>
      <li> This class is used for a comparison of CPU time.
      <li> Used to compare with and without SF BC.
      <li> Used to compare between Wilson and clover fermion.
      <li> Used to compare between single and parallel.
      <li> The Iwasaki gauge action is adopted.
      <li> The implementations are:
      <ul>
        <li> Nf=2 HMC with Wilson fermion
        <li> Nf=2 HMC with clover fermion
        <li> Nf=3 RHMC with Wilson fermion
        <li> Nf=3 RHMC with clover fermion
      </ul>
      <li> The massless limit shall be tuned with \f$\kappa_{ud}=\kappa_s\f$.
      <li> The Iwasaki gauge action is adopted.
      <li> The implementations are:
    </ul>
    (Coding history will be recovered from trac.)
    YAML is implemented.            [14 Nov 2012 Y.Namekawa]
    Selectors are implemented.      [03 Mar 2013 Y.Namekawa]
    (Selectors are replaced with factories by Aoyama-san)
*/

namespace Test_HMC_Clover_SF {
  //- test-private parameters
  namespace {
    const std::string filename_input  = "test_HMC_Clover_SF_Leapfrog_Nf2.yaml";
    const std::string filename_output = "stdout";

    class Parameters_Test_HMC_Clover_SF : public Parameters {
     public:
      Parameters_Test_HMC_Clover_SF()
      {
        Register_string("gauge_config_status", "NULL");

        Register_string("gauge_config_type_input", "NULL");
        Register_string("config_filename_input", "NULL");

        Register_string("gauge_config_type_output", "NULL");
        Register_string("config_filename_output", "NULL");

        Register_int("trajectory_number", 0);
        Register_int("trajectory_number_step", 0);
        Register_int("save_config_interval", 0);

        Register_string("verbose_level", "NULL");

        Register_double("expected_result", 0.0);
      }
    };
  }

  //- prototype declaration
  int leapfrog_Nf2(void);

#ifdef USE_TESTMANAGER_AUTOREGISTER
  namespace {
    static const bool is_registered = TestManager::RegisterTest(
      "HMC.Clover_SF.Leapfrog_Nf2",
      leapfrog_Nf2
      );
  }
#endif

  //====================================================================
  int leapfrog_Nf2(void)
  {
    // #####  parameter setup  #####
    int Nc   = CommonParameters::Nc();
    int Nvol = CommonParameters::Nvol();
    int Ndim = CommonParameters::Ndim();
    int NinG = 2 * Nc * Nc;

    Parameters *params_test       = new Parameters_Test_HMC_Clover_SF;
    Parameters *params_action_G   = ParametersFactory::New("Action.G_Rectangle_SF");
    Parameters *params_clover     = ParametersFactory::New("Fopr.Clover_SF");
    Parameters *params_proj       = ParametersFactory::New("Projection");
    Parameters *params_smear      = ParametersFactory::New("Smear");
    Parameters *params_dr_smear   = ParametersFactory::New("Director_Smear");
    Parameters *params_integrator = ParametersFactory::New("Builder_Integrator");
    Parameters *params_hmc        = ParametersFactory::New("HMC.General");

    Parameters *params_all = new Parameters;

    params_all->Register_Parameters("Test_HMC_Clover_SF", params_test);
    params_all->Register_Parameters("Action_G_Rectangle_SF", params_action_G);
    params_all->Register_Parameters("Fopr_Clover_Nf2_SF", params_clover);
    params_all->Register_Parameters("Projection", params_proj);
    params_all->Register_Parameters("Smear", params_smear);
    params_all->Register_Parameters("Director_Smear", params_dr_smear);
    params_all->Register_Parameters("Builder_Integrator", params_integrator);
    params_all->Register_Parameters("HMC_General", params_hmc);

    ParameterManager_YAML params_manager;
    params_manager.read_params(filename_input, params_all);

    const string str_gconf_status = params_test->get_string("gauge_config_status");
    const string str_gconf_read   = params_test->get_string("gauge_config_type_input");
    const string readfile         = params_test->get_string("config_filename_input");
    const string str_gconf_write  = params_test->get_string("gauge_config_type_output");
    const string writefile        = params_test->get_string("config_filename_output");
    int          iconf            = params_test->get_int("trajectory_number");
    int          Ntraj            = params_test->get_int("trajectory_number_step");
    const int    i_save_conf      = params_test->get_int("save_config_interval");
    const string str_vlevel       = params_test->get_string("verbose_level");
#ifdef USE_TEST
    const double expected_result = params_test->get_double("expected_result");
#endif

    const string str_gmset_type = params_clover->get_string("gamma_matrix_type");
    const string str_proj_type  = params_proj->get_string("projection_type");
    const string str_smear_type = params_smear->get_string("smear_type");

    Bridge::VerboseLevel vl = vout.set_verbose_level(str_vlevel);

    //- print input parameters
    vout.general(vl, "  gconf_status = %s\n", str_gconf_status.c_str());
    vout.general(vl, "  gconf_read  = %s\n", str_gconf_read.c_str());
    vout.general(vl, "  readfile    = %s\n", readfile.c_str());
    vout.general(vl, "  gconf_write = %s\n", str_gconf_write.c_str());
    vout.general(vl, "  writefile   = %s\n", writefile.c_str());
    vout.general(vl, "  iconf       = %d\n", iconf);
    vout.general(vl, "  Ntraj       = %d\n", Ntraj);
    vout.general(vl, "  i_save_conf = %d\n", i_save_conf);
    vout.general(vl, "  vlevel      = %s\n", str_vlevel.c_str());

    vout.general(vl, "  gmset_type  = %s\n", str_gmset_type.c_str());
    vout.general(vl, "  proj_type   = %s\n", str_proj_type.c_str());
    vout.general(vl, "  smear_type  = %s\n", str_smear_type.c_str());
    vout.general(vl, "\n");

    //- input parameter check
    int err = 0;
    err += ParameterCheck::non_NULL(str_gconf_read);
    err += ParameterCheck::non_NULL(readfile);
    err += ParameterCheck::non_NULL(str_gconf_write);
    err += ParameterCheck::non_NULL(writefile);
    err += ParameterCheck::non_zero(iconf);
    err += ParameterCheck::non_zero(Ntraj);
    err += ParameterCheck::non_zero(i_save_conf);

    if (err) {
      vout.crucial(vl, "Test_HMC_Clover_SF: Input parameters have not been set.\n");
      abort();
    }


    // #####  object setup  #####
    Field_G     *U          = new Field_G(Nvol, Ndim);
    GaugeConfig *gconf_read = new GaugeConfig(str_gconf_read);

    if (str_gconf_status == "Continue") {
      gconf_read->read_file((Field *)U, readfile);
    } else if (str_gconf_status == "Start_cold") {
      gconf_read->set_cold((Field *)U);
    } else {
      vout.crucial(vl, "Test_HMC_Clover_SF: unsupported gconf status \"%s\".\n", str_gconf_status.c_str());
      abort();
    }


    Action_G_Rectangle_SF *action_G = new Action_G_Rectangle_SF;
    action_G->set_parameters(*params_action_G);

    Fopr_Clover_SF *fopr_w = new    Fopr_Clover_SF();
    fopr_w->set_parameters(*params_clover);
    Force_F_Clover_SF *force_fopr_w = new Force_F_Clover_SF();
    force_fopr_w->set_parameters(*params_clover);


    Projection *proj  = Projection::New(str_proj_type);
    Smear      *smear = Smear::New(str_smear_type, proj);
    smear->set_parameters(*params_smear);

    ForceSmear *force_smear = ForceSmear::New(str_smear_type, proj);
    force_smear->set_parameters(*params_smear);

    Director_Smear *dr_smear = new Director_Smear((Smear *)smear);
    dr_smear->set_parameters(*params_dr_smear);

    Fopr_Smeared    *fopr_smear = new Fopr_Smeared((Fopr *)fopr_w, dr_smear);
    Force_F_Smeared *force_fopr_smear
      = new Force_F_Smeared((Force *)force_fopr_w, (ForceSmear *)force_smear, dr_smear);


    Action_F_Standard_SF *action_F_Nf2
      = new Action_F_Standard_SF((Fopr *)fopr_smear, (Force *)force_fopr_smear);
    action_F_Nf2->set_label("Nf=2 smeared clover fermion");


    valarray<Action *> actions(2);
    actions[0] = (Action *)action_F_Nf2;
    actions[1] = (Action *)action_G;

    valarray<Director *> directors(1);
    directors[0] = (Director *)dr_smear;

    Builder_Integrator *builder = new Builder_Integrator(actions, directors);
    builder->set_parameters(*params_integrator);
    Integrator *integrator = builder->build();


    //- M series random number intialization
    // int ndelay = 200000;
    // RandomNumbers_Mseries rand(ndelay);

    //- Mersenne Twister for random number generator.
    RandomNumbers_MT19937 *rand;
    if (iconf == 0) {
      std::vector<unsigned long> seed(4);
      seed[0] = 0x6a92;
      seed[1] = 0x3708;
      seed[2] = 0xab41;
      seed[3] = 0x5c52;
      rand    = new RandomNumbers_MT19937(seed);
    } else {
      rand = new RandomNumbers_MT19937(iconf);
    }

    HMC_General hmc(actions, directors, integrator, (RandomNumbers *)rand);
    hmc.set_parameters(*params_hmc);


    // ####  Execution main part  ####
    vout.general(vl, "HMC start: Ntraj = %d\n", Ntraj);

    double result = 0.0;
    for (int traj = 0; traj < Ntraj; ++traj) {
      vout.general(vl, "\n");
      vout.general(vl, "traj = %d\n", traj);

      result = hmc.update(*U);
    }

    if (Communicator::nodeid() == 0) {
      rand->writefile("rand_MT");
    }


    // #####  tidy up  #####
    delete params_test;
    delete params_action_G;
    delete params_clover;
    delete params_smear;
    delete params_dr_smear;
    delete params_integrator;
    delete params_hmc;
    delete params_all;

    delete U;
    delete gconf_read;

    delete rand;

    delete action_G;

    delete       fopr_w;
    delete force_fopr_w;
    delete action_F_Nf2;

    delete proj;
    delete           smear;
    delete     force_smear;
    delete         dr_smear;
    delete       fopr_smear;
    delete force_fopr_smear;

    delete builder;


#ifdef USE_TEST
    return Test::verify(expected_result, result);

#else
    return EXIT_SUCCESS;
#endif
  }
} // namespace Test_HMC_Clover_SF
