/*!
        @file    $Id: test_HMC_Quenched_Leapfrog.cpp #$

        @brief

        @author  Hideo Matsufuru <hideo.matsufuru@kek.jp> (matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate: 2013-04-27 12:28:50 #$

        @version $LastChangedRevision: 936 $
*/

#include "parameterManager_YAML.h"
#include "parameters_factory.h"

#include "bridgeIO.h"
using Bridge::vout;

#include "gaugeConfig.h"

#include "action_G_Plaq.h"
#include "action_G_Rectangle.h"

#include "randomNumbers_Mseries.h"

#include "hmc_Leapfrog.h"

#ifdef USE_TEST
#include "test.h"
#endif

#ifdef USE_TESTMANAGER_AUTOREGISTER
#include "testManager.h"
#endif

//====================================================================
//! Test of quenched HMC update.

/*!
    Test of quenched HMC with leapfrog integrator.
                                 [12 Apr 2012 H.Matsufuru]
    (Coding history will be recovered from trac.)
    YAML is implemented.         [14 Nov 2012 Y.Namekawa]
 */

namespace Test_HMC_Quenched {
  //- test-private parameters
  namespace {
    const std::string filename_input  = "test_HMC_Quenched_Leapfrog.yaml";
    const std::string filename_output = "stdout";

    class Parameters_Test_HMC_Quenched : public Parameters {
     public:
      Parameters_Test_HMC_Quenched()
      {
        Register_string("gauge_config_status", "NULL");

        Register_string("gauge_config_type_input", "NULL");
        Register_string("config_filename_input", "NULL");

        Register_string("gauge_config_type_output", "NULL");
        Register_string("config_filename_output", "NULL");

        Register_int("trajectory_number", 0);
        Register_int("trajectory_number_step", 0);
        Register_int("save_config_interval", 0);

        Register_string("verbose_level", "NULL");

        Register_double("expected_result", 0.0);
      }
    };
  }

  //- prototype declaration
  int leapfrog(void);

#ifdef USE_TESTMANAGER_AUTOREGISTER
  namespace {
    static const bool is_registered = TestManager::RegisterTest(
      "HMC.Quenched.Leapfrog",
      leapfrog
      );
  }
#endif

  //====================================================================
  int leapfrog(void)
  {
    // ####  parameter setup  ####
    int Nc   = CommonParameters::Nc();
    int Nvol = CommonParameters::Nvol();
    int Ndim = CommonParameters::Ndim();
    int NinG = 2 * Nc * Nc;

    Parameters *params_test     = new Parameters_Test_HMC_Quenched;
    Parameters *params_action_G = ParametersFactory::New("Action.G_Rectangle");
    Parameters *params_hmc      = ParametersFactory::New("HMC.Leapfrog");

    Parameters *params_all = new Parameters;

    params_all->Register_Parameters("Test_HMC_Quenched", params_test);
    params_all->Register_Parameters("Action_G_Rectangle", params_action_G);
    params_all->Register_Parameters("HMC_Leapfrog", params_hmc);

    ParameterManager_YAML params_manager;
    params_manager.read_params(filename_input, params_all);

    const string str_gconf_status = params_test->get_string("gauge_config_status");
    const string str_gconf_read   = params_test->get_string("gauge_config_type_input");
    const string readfile         = params_test->get_string("config_filename_input");
    const string str_gconf_write  = params_test->get_string("gauge_config_type_output");
    const string writefile        = params_test->get_string("config_filename_output");
    int          iconf            = params_test->get_int("trajectory_number");
    int          Ntraj            = params_test->get_int("trajectory_number_step");
    const int    i_save_conf      = params_test->get_int("save_config_interval");
    const string str_vlevel       = params_test->get_string("verbose_level");
#ifdef USE_TEST
    const double expected_result = params_test->get_double("expected_result");
#endif

    Bridge::VerboseLevel vl = vout.set_verbose_level(str_vlevel);

    //- print input parameters
    vout.general(vl, "  gconf_status = %s\n", str_gconf_status.c_str());
    vout.general(vl, "  gconf_read   = %s\n", str_gconf_read.c_str());
    vout.general(vl, "  readfile     = %s\n", readfile.c_str());
    vout.general(vl, "  gconf_write  = %s\n", str_gconf_write.c_str());
    vout.general(vl, "  writefile    = %s\n", writefile.c_str());
    vout.general(vl, "  iconf        = %d\n", iconf);
    vout.general(vl, "  Ntraj        = %d\n", Ntraj);
    vout.general(vl, "  i_save_conf  = %d\n", i_save_conf);
    vout.general(vl, "  vlevel       = %s\n", str_vlevel.c_str());
    vout.general(vl, "\n");

    //- input parameter check
    int err = 0;
    err += ParameterCheck::non_NULL(str_gconf_read);
    err += ParameterCheck::non_NULL(readfile);
    err += ParameterCheck::non_NULL(str_gconf_write);
    err += ParameterCheck::non_NULL(writefile);
    err += ParameterCheck::non_zero(iconf);
    err += ParameterCheck::non_zero(Ntraj);
    err += ParameterCheck::non_zero(i_save_conf);

    if (err) {
      vout.crucial(vl, "Test_HMC_Quenched: Input parameters have not been set.\n");
      abort();
    }


    // ####  object setup  ####
    Field_G     *U          = new Field_G(Nvol, Ndim);
    GaugeConfig *gconf_read = new GaugeConfig(str_gconf_read);

    if (str_gconf_status == "Continue") {
      gconf_read->read_file((Field *)U, readfile);
    } else if (str_gconf_status == "Start_cold") {
      gconf_read->set_cold((Field *)U);
    } else {
      vout.crucial(vl, "Test_HMC_Quenched: unsupported gconf status \"%s\".\n", str_gconf_status.c_str());
      abort();
    }


    Action_G_Rectangle *action_G = new Action_G_Rectangle;
    action_G->set_parameters(*params_action_G);

    RandomNumbers *rand = new RandomNumbers_Mseries(iconf);

    std::valarray<Action *> actions(1);
    actions[0] = (Action *)action_G;

    HMC_Leapfrog hmc(actions, (RandomNumbers *)rand);
    hmc.set_parameters(*params_hmc);


    // ####  Execution main part  ####
    vout.general(vl, "HMC: Ntraj = %d\n", Ntraj);

    double result = 0.0;
    for (int traj = 0; traj < Ntraj; ++traj) {
      vout.general(vl, "\n");
      vout.general(vl, "traj = %d\n", traj);

      result = hmc.update(*U);
    }


    // ####  tidy up  ####
    delete params_test;
    delete params_action_G;
    delete params_hmc;
    delete params_all;

    delete U;
    delete gconf_read;

    delete rand;
    delete action_G;


#ifdef USE_TEST
    return Test::verify(expected_result, result);

#else
    return EXIT_SUCCESS;
#endif
  }
} // namespace Test_HMC_Quenched
