/*!
        @file    $Id: test_IO_Data_Text.cpp #$

        @brief

        @author  $LastChangedBy: namekawa $

        @date    $LastChangedDate: 2013-03-21 15:28:34 #$

        @version $LastChangedRevision: 889 $
*/

#include "parameterManager_YAML.h"

#include "bridgeIO.h"
using Bridge::vout;

#include "gaugeConfig.h"

#include "dataIO_Text.h"
#include "bridge_complex.h"

#ifdef USE_TEST
#include "test.h"
#endif

#ifdef USE_TESTMANAGER_AUTOREGISTER
#include "testManager.h"
#endif

//====================================================================
//! Test of I/O.

/*!
    (Implemented by Aoyama-san.)
    (Coding history will be recovered from trac.)
    YAML is implemented.      [14 Nov 2012 Y.Namekawa]
*/

namespace Test_IO_Data {
  //- test-private parameters
  namespace {
    const std::string filename_input  = "test_IO_Data_Text.yaml";
    const std::string filename_output = "stdout";

    class Parameters_Test_IO_GaugeConfig : public Parameters {
     public:
      Parameters_Test_IO_GaugeConfig()
      {
        Register_string("gauge_config_type_input", "NULL");
        Register_string("config_filename_input", "NULL");
        Register_string("config_filename_output", "NULL");

        Register_int("data_size", 1024);

        Register_string("verbose_level", "NULL");
        Register_double("expected_result", 0.0);
      }
    };
  }

  //- prototype declaration
  int test_io_data_text(void);

#ifdef USE_TESTMANAGER_AUTOREGISTER
  namespace {
    static const bool is_registered = TestManager::RegisterTest(
      "IO.Data.Text",
      test_io_data_text
      );
  }
#endif

  //====================================================================
  int test_io_data_text(void)
  {
    // ####  parameter setup  ####
    int          Ndim = CommonParameters::Ndim();
    int          Nvol = CommonParameters::Nvol();
    const double tiny = CommonParameters::epsilon_criterion();

    Parameters_Test_IO_GaugeConfig params_test;

    Parameters params_all;

    params_all.Register_Parameters("Test_IO_Data", &params_test);

    ParameterManager_YAML params_manager;
    params_manager.read_params(filename_input, &params_all);

    const string str_gconf_read = params_test.get_string("gauge_config_type_input");
    const string readfile       = params_test.get_string("config_filename_input");
    const string testfile       = params_test.get_string("config_filename_output");
    const int    data_size      = params_test.get_int("data_size");
    const string str_vlevel     = params_test.get_string("verbose_level");
#ifdef USE_TEST
    const double expected_result = params_test.get_double("expected_result");
#endif

    Bridge::VerboseLevel vl = vout.set_verbose_level(str_vlevel);

    //- print input parameters
    vout.general(vl, "  gconf_read = %s\n", str_gconf_read.c_str());
    vout.general(vl, "  readfile   = %s\n", readfile.c_str());
    vout.general(vl, "  testfile   = %s\n", testfile.c_str());
    vout.general(vl, "  data_size  = %d\n", data_size);
    vout.general(vl, "  vlevel     = %s\n", str_vlevel.c_str());
    vout.general(vl, "\n");

    //- input parameter check
    int err = 0;
    err += ParameterCheck::non_NULL(str_gconf_read);
    err += ParameterCheck::non_NULL(readfile);
    err += ParameterCheck::non_NULL(testfile);

    if (err) {
      vout.crucial(vl, "Test_IO_Data: Input parameters have not been set.\n");
      abort();
    }


    // ####  Set up a gauge configuration  ####
    Field_G     *U          = new Field_G(Nvol, Ndim);
    GaugeConfig *gconf_read = new GaugeConfig(str_gconf_read);
    gconf_read->read_file((Field *)U, readfile);


    // #####  object setup  #####
    DataIO *data_io = new DataIO_Text;


    // ####  Execution main part  ####
    err = 0;

    //- first write to file.
    std::valarray<double> array(data_size);
    double                *p = U->ptr(0);

    for (size_t i = 0; i < data_size; ++i) {
      array[i] = *p++;
    }

    data_io->write_file(array, testfile, false);


    //- write again with complex data.
    std::valarray<dcomplex> arrayc(data_size);
    p = U->ptr(0);

    for (size_t i = 0; i < data_size; ++i) {
      double x = *p++;
      double y = *p++;

      arrayc[i] = cmplx(x, y);
    }

    data_io->write_file(arrayc, testfile, true);


    //- then read back from file.
    std::valarray<double> array2(data_size);

    data_io->read_file(array2, testfile);


    //- check with reference config.
    for (size_t i = 0; i < data_size; ++i) {
      if (abs(array[i] - array2[i]) > tiny) ++err;
    }

    vout.general(vl, "Test_IO_Data:\t%s\n", (err == 0) ? "ok" : "failed");


    int result = err;


    // ####  tydy up  ####
    delete gconf_read;
    delete U;
    delete data_io;


#ifdef USE_TEST
    return Test::verify(expected_result, result);

#else
    return EXIT_SUCCESS;
#endif
  }
} // namespace Test_IO_Data
