/*!
        @file    $Id:: test_RandomNumbers_Mseries_Uniform.cpp #$

        @brief

        @author  Hideo Matsufuru <hideo.matsufuru@kek.jp> (matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-19 14:15:23 #$

        @version $LastChangedRevision: 936 $
*/

#include "parameterManager_YAML.h"

#include "bridgeIO.h"
using Bridge::vout;

#include "randomNumbers_Mseries.h"

#ifdef USE_TEST
#include "test.h"
#endif

#ifdef USE_TESTMANAGER_AUTOREGISTER
#include "testManager.h"
#endif

//====================================================================
//! Test of random number generator.

/*!
                                [28 Dec 2011 H.Matsufuru]
    (Coding history will be recovered from trac.)
    YAML is implemented.        [14 Nov 2012 Y.Namekawa]
 */

namespace Test_RandomNumbers {
  //- test-private parameters
  namespace {
    const std::string filename_input  = "test_RandomNumbers_Mseries_Uniform.yaml";
    const std::string filename_output = "stdout";

    class Parameters_Test_RandomNumbers : public Parameters {
     public:
      Parameters_Test_RandomNumbers()
      {
        Register_int("number_of_seeds", 0);
        Register_int("seed_base", 0);
        Register_int("number_of_samples", 0);

        Register_string("verbose_level", "NULL");

        Register_double("expected_result", 0.0);
      }
    };
  }

  //- prototype declaration
  int uniform_calc_pi(void);

#ifdef USE_TESTMANAGER_AUTOREGISTER
  namespace {
    static const bool is_registered = TestManager::RegisterTest(
      "RandomNumbers.Mseries.Uniform",
      uniform_calc_pi
      );
  }
#endif

  //====================================================================
  int uniform_calc_pi(void)
  {
    // ####  parameter setup  ####
    Parameters_Test_RandomNumbers params_test;

    Parameters params_all;

    params_all.Register_Parameters("Test_RandomNumbers", &params_test);

    ParameterManager_YAML params_manager;
    params_manager.read_params(filename_input, &params_all);

    int          Nseed      = params_test.get_int("number_of_seeds");
    int          seed_base  = params_test.get_int("seed_base");
    int          Nrand      = params_test.get_int("number_of_samples");
    const string str_vlevel = params_test.get_string("verbose_level");
#ifdef USE_TEST
    const double expected_result = params_test.get_double("expected_result");
#endif

    Bridge::VerboseLevel vl = vout.set_verbose_level(str_vlevel);

    //- print input parameters
    vout.general(vl, "  Nseed     = %d\n", Nseed);
    vout.general(vl, "  seed_base = %d\n", seed_base);
    vout.general(vl, "  Nrand     = %d\n", Nrand);
    vout.general(vl, "  vlevel    = %s\n", str_vlevel.c_str());
    vout.general(vl, "\n");


    // ####  Execution main part  ####
    vout.general(vl, "\n");
    vout.general(vl, "Monte Calro estimate of pi:\n");
    vout.general(vl, "  number of samples = %10d\n", Nrand);
    vout.general(vl, "        seed    estimate of pi\n");

    double t1 = 0.0;
    double t2 = 0.0;
    for (int iseed = 0; iseed < Nseed; ++iseed) {
      int iseed2 = seed_base + iseed;

      RandomNumbers *rand = new RandomNumbers_Mseries(iseed2);

      int Npi = 0;
      for (int i = 0; i < Nrand; ++i) {
        double rn1 = rand->get();
        double rn2 = rand->get();
        double r   = rn1 * rn1 + rn2 * rn2;
        if (r < 1.0) { ++Npi;  }
        //  vout.general(vl, "  %10.8f  %10.8f\n",rn1,rn2);
      }

      double pi_exp = (4.0 * Npi) / Nrand;

      t1 += pi_exp;
      t2 += pi_exp * pi_exp;

      //vout.general(vl, "  estimate of pi    = %10.8f\n",pi_exp);
      vout.general(vl, "  %10d    %14.10f\n", iseed2, pi_exp);

      delete rand;
    }

    double api = t1 / (double)Nseed;
    double vpi = t2 / (double)Nseed - api * api;
    double dpi = sqrt(vpi);
    double epi = dpi / sqrt((double)Nseed - 1);

    double pi = 3.141592653589793;
    vout.general(vl, "  true value = %10.8f\n", pi);
    vout.general(vl, "  average    = %10.8f\n", api);
    vout.general(vl, "  variance   = %10.8f\n", vpi);
    vout.general(vl, "  deviation  = %10.8f\n", dpi);
    vout.general(vl, "  error      = %10.8f\n", epi);

    double result = epi;


#ifdef USE_TEST
    return Test::verify(expected_result, result);

#else
    return EXIT_SUCCESS;
#endif
  }
} // namespace Test_RandomNumbers
