/*!
        @file    $Id: test_Rational_Approx.cpp #$

        @brief

        @author  Hideo Matsufuru <hideo.matsufuru@kek.jp> (matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#include "parameterManager_YAML.h"

#include "bridgeIO.h"
using Bridge::vout;

#include "math_Rational.h"

#ifdef USE_TEST
#include "test.h"
#endif

#ifdef USE_TESTMANAGER_AUTOREGISTER
#include "testManager.h"
#endif

//====================================================================
//! Test of rational approximation of fermion operators.

/*!
                                [28 Dec 2011 H.Matsufuru]
    (Coding history will be recovered from trac.)
    YAML is implemented.        [14 Nov 2012 Y.Namekawa]
 */

namespace Test_Rational {
  //- test-private parameters
  namespace {
    const std::string filename_input  = "test_Rational_Approx.yaml";
    const std::string filename_output = "stdout";

    class Parameters_Test_Rational : public Parameters {
     public:
      Parameters_Test_Rational()
      {
        Register_string("verbose_level", "NULL");

        Register_double("expected_result", 0.0);
      }
    };
  }

  //- prototype declaration
  int approx(void);

#ifdef USE_TESTMANAGER_AUTOREGISTER
  namespace {
    static const bool is_registered = TestManager::RegisterTest(
      "Rational.Approx",
      approx
      );
  }
#endif

  //====================================================================
  int approx(void)
  {
    // ####  parameter setup  ####
    Parameters_Test_Rational params_test;
    Parameters_Math_Rational params_rational;

    Parameters params_all;

    params_all.Register_Parameters("Test_Rational", &params_test);
    params_all.Register_Parameters("Math_Rational", &params_rational);

    ParameterManager_YAML params_manager;
    params_manager.read_params(filename_input, &params_all);

    const string str_vlevel = params_test.get_string("verbose_level");
#ifdef USE_TEST
    const double expected_result = params_test.get_double("expected_result");
#endif

    const int    n_exp  = params_rational.get_int("exponent_numerator");
    const int    d_exp  = params_rational.get_int("exponent_denominator");
    const double x_min  = params_rational.get_double("lower_bound");
    const double x_max  = params_rational.get_double("upper_bound");
    const int    Npoint = params_rational.get_int("number_of_partitions");

    Bridge::VerboseLevel vl = vout.set_verbose_level(str_vlevel);

    //- print input parameters
    vout.general(vl, "  vlevel = %s\n", str_vlevel.c_str());
    vout.general(vl, "  n_exp  = %d\n", n_exp);
    vout.general(vl, "  d_exp  = %d\n", d_exp);
    vout.general(vl, "  x_min  = %12.8f\n", x_min);
    vout.general(vl, "  x_max  = %12.8f\n", x_max);
    vout.general(vl, "  Npoint = %d\n", Npoint);
    vout.general(vl, "\n");


    // ####  object setup  ####
    Math_Rational rational;
    rational.set_parameters(params_rational);


    // ####  Execution main part  ####
    double x     = x_min;
    double Dx    = (x_max - x_min) / Npoint;
    double p_exp = ((double)n_exp) / ((double)d_exp);

    for (int j = 0; j < Npoint + 1; ++j) {
      double y1 = rational.func(x);
      double y2 = pow(x, p_exp);

      vout.general(vl, "x,rational,x^(n/d) = %12.8f  %12.8f  %12.8f \n", x, y1, y2);

      x += Dx;
    }

    double result = x;


#ifdef USE_TEST
    return Test::verify(expected_result, result);

#else
    return EXIT_SUCCESS;
#endif
  }
} // namespace Test_Rational
