/*!
        @file    $Id: test_Rational_Inverse.cpp #$

        @brief

        @author  Hideo Matsufuru <hideo.matsufuru@kek.jp> (matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-19 14:15:23 #$

        @version $LastChangedRevision: 936 $
*/

#include "parameterManager_YAML.h"
#include "parameters_factory.h"

#include "bridgeIO.h"
using Bridge::vout;

#include "gaugeConfig.h"

#include "fopr_Clover.h"
#include "fopr_Rational.h"

#include "source.h"

#ifdef USE_TEST
#include "test.h"
#endif

#ifdef USE_TESTMANAGER_AUTOREGISTER
#include "testManager.h"
#endif

//====================================================================
//! Test of rational approximation of fermion operators.

/*!
                                [28 Dec 2011 H.Matsufuru]
    (Coding history will be recovered from trac.)
    YAML is implemented.        [14 Nov 2012 Y.Namekawa]
    Selectors are implemented.  [03 Mar 2013 Y.Namekawa]
    (Selectors are replaced with factories by Aoyama-san)
 */

namespace Test_Rational {
  //- test-private parameters
  namespace {
    const std::string filename_input  = "test_Rational_Inverse.yaml";
    const std::string filename_output = "stdout";

    class Parameters_Test_Rational : public Parameters {
     public:
      Parameters_Test_Rational()
      {
        Register_string("gauge_config_type_input", "NULL");
        Register_string("config_filename_input", "NULL");

        Register_string("verbose_level", "NULL");

        Register_double("expected_result", 0.0);
      }
    };
  }

  //- prototype declaration
  int inverse(void);

#ifdef USE_TESTMANAGER_AUTOREGISTER
  namespace {
    static const bool is_registered = TestManager::RegisterTest(
      "Rational.Inverse",
      inverse
      );
  }
#endif

  //====================================================================
  int inverse(void)
  {
    // #####  parameter setup  #####
    int Ndim = CommonParameters::Ndim();
    int Nc   = CommonParameters::Nc();
    int Nd   = CommonParameters::Nd();
    int Nvol = CommonParameters::Nvol();
    int NinF = 2 * Nc * Nd;

    Parameters *params_test     = new Parameters_Test_Rational;
    Parameters *params_clover   = ParametersFactory::New("Fopr.Clover");
    Parameters *params_rational = ParametersFactory::New("Fopr.Rational");
    Parameters *params_source   = ParametersFactory::New("Source");

    Parameters *params_all = new Parameters;

    params_all->Register_Parameters("Test_Rational", params_test);
    params_all->Register_Parameters("Fopr_Clover", params_clover);
    params_all->Register_Parameters("Fopr_Rational", params_rational);
    params_all->Register_Parameters("Source", params_source);

    ParameterManager_YAML params_manager;
    params_manager.read_params(filename_input, params_all);

    const string str_gconf_read = params_test->get_string("gauge_config_type_input");
    const string readfile       = params_test->get_string("config_filename_input");
    const string str_vlevel     = params_test->get_string("verbose_level");
#ifdef USE_TEST
    const double expected_result = params_test->get_double("expected_result");
#endif

    const string str_gmset_type  = params_clover->get_string("gamma_matrix_type");
    const string str_source_type = params_source->get_string("source_type");

    Bridge::VerboseLevel vl = vout.set_verbose_level(str_vlevel);

    //- print input parameters
    vout.general(vl, "  gconf_read  = %s\n", str_gconf_read.c_str());
    vout.general(vl, "  readfile    = %s\n", readfile.c_str());
    vout.general(vl, "  vlevel      = %s\n", str_vlevel.c_str());
    vout.general(vl, "  gmset_type  = %s\n", str_gmset_type.c_str());
    vout.general(vl, "  source_type = %s\n", str_source_type.c_str());
    vout.general(vl, "\n");

    //- input parameter check
    int err = 0;
    err += ParameterCheck::non_NULL(str_gconf_read);
    err += ParameterCheck::non_NULL(readfile);

    if (err) {
      vout.crucial(vl, "Test_Rational: Input parameters have not been set.\n");
      abort();
    }


    // ####  object setup  ####
    Field_G     *U          = new Field_G(Nvol, Ndim);
    GaugeConfig *gconf_read = new GaugeConfig(str_gconf_read);
    gconf_read->read_file((Field *)U, readfile);
    // gconf_read->set_cold((Field*)U);

    Fopr_Clover *fopr_c = new Fopr_Clover(str_gmset_type);
    fopr_c->set_parameters(*params_clover);
    fopr_c->set_config(U);

    Fopr_Rational *fopr_r = new Fopr_Rational((Fopr *)fopr_c);
    fopr_r->set_parameters(*params_rational);

    Source *source = Source::New(str_source_type);
    source->set_parameters(*params_source);


    // ####  Execution main part  ####
    Field   xq(NinF, Nvol, 1), b(NinF, Nvol, 1);
    Field   v(NinF, Nvol, 1), w(NinF, Nvol, 1);
    Field_F b2;

    double vv;
    {
      int ispin = 0;
      {
        int icolor = 0;
        //for(int ispin = 0; ispin < Nd; ++ispin){
        //    for(int icolor = 0; icolor < Nc; ++icolor){

        int idx = icolor + Nc * ispin;
        source->set(b2, idx);

        b = (Field)b2;

        v = fopr_r->mult(b);

        fopr_c->set_mode("DdagD");
        w = fopr_c->mult(v);

        v  = fopr_r->mult(w);
        v -= b;
        vv = v.norm2();

        vout.general(vl, "  standard norm2 = %.8e\n", vv);
      }
    }

    double result = vv;


    // #####  tidy up  #####
    delete params_test;
    delete params_clover;
    delete params_rational;
    delete params_source;
    delete params_all;

    delete U;
    delete gconf_read;

    delete source;
    delete fopr_r;
    delete fopr_c;


#ifdef USE_TEST
    return Test::verify(expected_result, result);

#else
    return EXIT_SUCCESS;
#endif
  }
} // namespace Test_Rational
