/*!
        @file    $Id: test_Rational_Smeared.cpp #$

        @brief

        @author  Hideo Matsufuru <hideo.matsufuru@kek.jp> (matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-19 14:15:23 #$

        @version $LastChangedRevision: 936 $
*/

#include "parameterManager_YAML.h"
#include "parameters_factory.h"

#include "bridgeIO.h"
using Bridge::vout;

#include "gaugeConfig.h"

#include "fopr_Clover.h"
#include "fopr_Rational.h"

#include "director_Smear.h"
#include "fopr_Smeared.h"

#include "projection.h"
#include "smear.h"
#include "source.h"

#ifdef USE_TEST
#include "test.h"
#endif

#ifdef USE_TESTMANAGER_AUTOREGISTER
#include "testManager.h"
#endif

//====================================================================
//! Test of rational approximation of fermion operators.

/*!
                                [28 Dec 2011 H.Matsufuru]
    (Coding history will be recovered from trac.)
    YAML is implemented.        [14 Nov 2012 Y.Namekawa]
    Selectors are implemented.  [03 Mar 2013 Y.Namekawa]
    (Selectors are replaced with factories by Aoyama-san)
 */

namespace Test_Rational {
  //- test-private parameters
  namespace {
    const std::string filename_input  = "test_Rational_Smeared.yaml";
    const std::string filename_output = "stdout";

    class Parameters_Test_Rational : public Parameters {
     public:
      Parameters_Test_Rational()
      {
        Register_string("gauge_config_type_input", "NULL");
        Register_string("config_filename_input", "NULL");

        Register_string("verbose_level", "NULL");

        Register_double("expected_result", 0.0);
      }
    };
  }

  //- prototype declaration
  int smeared_rational(void);

#ifdef USE_TESTMANAGER_AUTOREGISTER
  namespace {
    static const bool is_registered = TestManager::RegisterTest(
      "Rational.Smeared_Rational",
      smeared_rational
      );
  }
#endif

  //====================================================================
  int smeared_rational(void)
  {
    // #####  parameter setup  #####
    int Ndim = CommonParameters::Ndim();
    int Nc   = CommonParameters::Nc();
    int Nd   = CommonParameters::Nd();
    int Nvol = CommonParameters::Nvol();
    int NinF = 2 * Nc * Nd;

    Parameters *params_test     = new Parameters_Test_Rational;
    Parameters *params_clover   = ParametersFactory::New("Fopr.Clover");
    Parameters *params_proj     = ParametersFactory::New("Projection");
    Parameters *params_smear    = ParametersFactory::New("Smear");
    Parameters *params_dr_smear = ParametersFactory::New("Director_Smear");
    Parameters *params_rational = ParametersFactory::New("Fopr.Rational");
    Parameters *params_source   = ParametersFactory::New("Source");

    Parameters *params_all = new Parameters;

    params_all->Register_Parameters("Test_Rational", params_test);
    params_all->Register_Parameters("Fopr_Clover", params_clover);
    params_all->Register_Parameters("Projection", params_proj);
    params_all->Register_Parameters("Smear", params_smear);
    params_all->Register_Parameters("Director_Smear", params_dr_smear);
    params_all->Register_Parameters("Fopr_Rational", params_rational);
    params_all->Register_Parameters("Source", params_source);

    ParameterManager_YAML params_manager;
    params_manager.read_params(filename_input, params_all);

    const string str_gconf_read = params_test->get_string("gauge_config_type_input");
    const string readfile       = params_test->get_string("config_filename_input");
    const string str_vlevel     = params_test->get_string("verbose_level");
#ifdef USE_TEST
    const double expected_result = params_test->get_double("expected_result");
#endif

    const string str_gmset_type  = params_clover->get_string("gamma_matrix_type");
    const string str_proj_type   = params_proj->get_string("projection_type");
    const string str_smear_type  = params_smear->get_string("smear_type");
    const string str_source_type = params_source->get_string("source_type");

    Bridge::VerboseLevel vl = vout.set_verbose_level(str_vlevel);

    //- print input parameters
    vout.general(vl, "  gconf_read  = %s\n", str_gconf_read.c_str());
    vout.general(vl, "  readfile    = %s\n", readfile.c_str());
    vout.general(vl, "  vlevel      = %s\n", str_vlevel.c_str());
    vout.general(vl, "  gmset_type  = %s\n", str_gmset_type.c_str());
    vout.general(vl, "  proj_type   = %s\n", str_proj_type.c_str());
    vout.general(vl, "  smear_type  = %s\n", str_smear_type.c_str());
    vout.general(vl, "  source_type = %s\n", str_source_type.c_str());
    vout.general(vl, "\n");

    //- input parameter check
    int err = 0;
    err += ParameterCheck::non_NULL(str_gconf_read);
    err += ParameterCheck::non_NULL(readfile);

    if (err) {
      vout.crucial(vl, "Test_Rational: Input parameters have not been set.\n");
      abort();
    }


    // #####  object setup  #####
    Field_G     *U          = new Field_G(Nvol, Ndim);
    GaugeConfig *gconf_read = new GaugeConfig(str_gconf_read);
    gconf_read->read_file((Field *)U, readfile);
    // gconf_read->set_cold((Field*)U);

    Fopr_Clover *fopr_c = new Fopr_Clover(str_gmset_type);
    fopr_c->set_parameters(*params_clover);

    Projection *proj  = Projection::New(str_proj_type);
    Smear      *smear = Smear::New(str_smear_type, proj);
    smear->set_parameters(*params_smear);

    Director_Smear *dr_smear = new Director_Smear((Smear *)smear);
    dr_smear->set_parameters(*params_dr_smear);

    Fopr_Rational *fopr_r = new Fopr_Rational((Fopr *)fopr_c);
    fopr_r->set_parameters(*params_rational);

    Source *source = Source::New(str_source_type);
    source->set_parameters(*params_source);

    //- Smeared-rational
    Fopr_Smeared fopr_smear_1((Fopr *)fopr_c, dr_smear);
    Fopr_Smeared fopr_smear_2((Fopr *)fopr_r, dr_smear);
    fopr_smear_1.set_config(U);
    fopr_smear_2.set_config(U);

    //- Rational-smeared
    Fopr_Rational fopr_r_smear_1((Fopr *)&fopr_smear_1);
    fopr_r_smear_1.set_parameters(*params_rational);
    fopr_r_smear_1.set_config(U);


    // ####  Execution main part  ####
    Field   xq(NinF, Nvol, 1), b(NinF, Nvol, 1);
    Field   v(NinF, Nvol, 1), w(NinF, Nvol, 1);
    Field_F b2;

    vout.general(vl, "Consistency check of smeared-rational and rational-smeared.\n");

    vout.general(vl, "Smeared-rational:\n");

    {
      int ispin = 0;
      {
        int icolor = 0;
        //for(int ispin = 0; ispin < Nd; ++ispin){
        //    for(int icolor = 0; icolor < Nc; ++icolor){

        int idx = icolor + Nc * ispin;
        source->set(b2, idx);

        b = (Field)b2;

        v = fopr_smear_2.mult(b);

        fopr_smear_1.set_mode("DdagD");
        w = fopr_smear_1.mult(v);

        v = fopr_smear_2.mult(w);

        v -= b;
        double vv = v.norm2();

        vout.general(vl, "  standard norm2 = %.8e\n", vv);
      }
    }

    vout.general(vl, "Rational-smeared:\n");

    double vv;
    {
      int ispin = 0;
      {
        int icolor = 0;
        //for(int ispin = 0; ispin < Nd; ++ispin){
        //    for(int icolor = 0; icolor < Nc; ++icolor){

        int idx = icolor + Nc * ispin;
        source->set(b2, idx);

        b = (Field)b2;

        v = fopr_r_smear_1.mult(b);

        fopr_smear_1.set_mode("DdagD");
        w = fopr_smear_1.mult(v);

        v = fopr_r_smear_1.mult(w);

        v -= b;
        vv = v.norm2();

        vout.general(vl, "  standard norm2 = %.8e\n", vv);
      }
    }

    double result = vv;


    // ####  tydy up  ####
    delete params_test;
    delete params_clover;
    delete params_proj;
    delete params_smear;
    delete params_dr_smear;
    delete params_rational;
    delete params_all;

    delete U;
    delete gconf_read;

    delete source;
    delete fopr_r;
    delete dr_smear;
    delete    smear;
    delete proj;
    delete fopr_c;


#ifdef USE_TEST
    return Test::verify(expected_result, result);

#else
    return EXIT_SUCCESS;
#endif
  }
} // namespace Test_Rational
