/*!
        @file    $Id:: test_SF_fAfP_Boundary_Meson_2ptFunction.cpp #$

        @brief

        @author  Yusuke Taniguchi <tanigchi@het.ph.tsukuba.ac.jp>
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-19 14:15:23 #$

        @version $LastChangedRevision: 936 $
*/

#include "parameterManager_YAML.h"
#include "parameters_factory.h"

#include "bridgeIO.h"
using Bridge::vout;

#include "gaugeConfig.h"

#include "fopr_Clover_SF.h"
#include "source_Wall_SF.h"

#include "director_Smear.h"
#include "fopr_Smeared.h"

#include "gammaMatrixSet.h"
#include "projection.h"
#include "smear.h"
#include "solver.h"

#include "fprop_Standard_lex.h"

#include "corr2pt_Wilson_SF.h"

#ifdef USE_TEST
#include "test.h"
#endif

#ifdef USE_TESTMANAGER_AUTOREGISTER
#include "testManager.h"
#endif

//====================================================================

/*!
    Calculate fA and fP for PCAC mass with SF BC.
    <ul>
      <li>The Dirac representaion of the gamma matrix is adopted.
      <li>Specified for
      <ul>
        <li>Number of smearing Nsmear=1 with parameter rho=0.1.
      </ul>
    </ul>
    (Coding history will be recovered from trac.)
    YAML is implemented.                  [14 Nov 2012 Y.Namekawa]
    Fprop and selectors are implemented.  [03 Mar 2013 Y.Namekawa]
    (Selectors are replaced with factories by Aoyama-san)
*/

namespace Test_SF_fAfP {
  //- test-private parameters
  namespace {
    const std::string filename_input  = "test_SF_fAfP_Boundary_Meson_2ptFunction.yaml";
    const std::string filename_output = "stdout";

    class Parameters_Test_SF_fAfP : public Parameters {
     public:
      Parameters_Test_SF_fAfP()
      {
        Register_string("gauge_config_type_input", "NULL");
        Register_string("config_filename_input", "NULL");

        Register_string("gauge_config_type_output", "NULL");
        Register_string("config_filename_output", "NULL");

        Register_string("verbose_level", "NULL");

        Register_double("expected_result", 0.0);
      }
    };
  }

  //- prototype declaration
  int boundary_meson_2ptFunction(void);

#ifdef USE_TESTMANAGER_AUTOREGISTER
  namespace {
    static const bool is_registered = TestManager::RegisterTest(
      "SF_fAfP.Boundary_Meson_2ptFunction",
      boundary_meson_2ptFunction
      );
  }
#endif

  //====================================================================
  int boundary_meson_2ptFunction(void)
  {
    // #####  parameter setup  #####
    int Nc   = CommonParameters::Nc();
    int Nd   = CommonParameters::Nd();
    int Ndim = CommonParameters::Ndim();
    int Nvol = CommonParameters::Nvol();

    Parameters *params_test     = new Parameters_Test_SF_fAfP;
    Parameters *params_clover   = ParametersFactory::New("Fopr.Clover_SF");
    Parameters *params_proj     = ParametersFactory::New("Projection");
    Parameters *params_smear    = ParametersFactory::New("Smear");
    Parameters *params_dr_smear = ParametersFactory::New("Director_Smear");
    Parameters *params_solver   = ParametersFactory::New("Solver");
    Parameters *params_source   = ParametersFactory::New("Source_Wall_SF");

    Parameters *params_all = new Parameters;

    params_all->Register_Parameters("Test_SF_fAfP", params_test);
    params_all->Register_Parameters("Fopr_Clover_SF", params_clover);
    params_all->Register_Parameters("Projection", params_proj);
    params_all->Register_Parameters("Smear_SF", params_smear);
    params_all->Register_Parameters("Director_Smear", params_dr_smear);
    params_all->Register_Parameters("Solver", params_solver);
    params_all->Register_Parameters("Source_Wall_SF", params_source);

    ParameterManager_YAML params_manager;
    params_manager.read_params(filename_input, params_all);

    const string str_gconf_read  = params_test->get_string("gauge_config_type_input");
    const string readfile        = params_test->get_string("config_filename_input");
    const string str_gconf_write = params_test->get_string("gauge_config_type_output");
    const string writefile       = params_test->get_string("config_filename_output");
    const string str_vlevel      = params_test->get_string("verbose_level");
#ifdef USE_TEST
    const double expected_result = params_test->get_double("expected_result");
#endif

    const string str_gmset_type  = params_clover->get_string("gamma_matrix_type");
    const string str_proj_type   = params_proj->get_string("projection_type");
    const string str_smear_type  = params_smear->get_string("smear_type");
    const string str_solver_type = params_solver->get_string("solver_type");

    Bridge::VerboseLevel vl = vout.set_verbose_level(str_vlevel);

    //- print input parameters
    vout.general(vl, "  gconf_read  = %s\n", str_gconf_read.c_str());
    vout.general(vl, "  readfile    = %s\n", readfile.c_str());
    vout.general(vl, "  gconf_write = %s\n", str_gconf_write.c_str());
    vout.general(vl, "  writefile   = %s\n", writefile.c_str());
    vout.general(vl, "  vlevel      = %s\n", str_vlevel.c_str());
    vout.general(vl, "  gmset_type  = %s\n", str_gmset_type.c_str());
    vout.general(vl, "  proj_type   = %s\n", str_proj_type.c_str());
    vout.general(vl, "  smear_type  = %s\n", str_smear_type.c_str());
    vout.general(vl, "  solver_type = %s\n", str_solver_type.c_str());
    vout.general(vl, "\n");

    //- input parameter check
    int err = 0;
    err += ParameterCheck::non_NULL(str_gconf_read);
    err += ParameterCheck::non_NULL(readfile);
    err += ParameterCheck::non_NULL(str_gconf_write);
    err += ParameterCheck::non_NULL(writefile);

    if (err) {
      vout.crucial(vl, "Test_SF_fAfP: Input parameters have not been set.\n");
      abort();
    }


    // ####  Set up a gauge configuration  ####
    Field_G     *U          = new Field_G(Nvol, Ndim);
    GaugeConfig *gconf_read = new GaugeConfig(str_gconf_read);
    gconf_read->read_file((Field *)U, readfile);
    // gconf_read->set_cold((Field*)U);

    Projection *proj = Projection::New(str_proj_type);

    Smear *smear = Smear::New(str_smear_type, proj);
    smear->set_parameters(*params_smear);

    Director_Smear *dr_smear = new Director_Smear((Smear *)smear);
    dr_smear->set_parameters(*params_dr_smear);
    dr_smear->set_config(U);

    int     Nsmear  = dr_smear->get_Nsmear();
    Field_G *Usmear = (Field_G *)dr_smear->getptr_smearedConfig(Nsmear);


    // ####  object setup  #####
    GammaMatrixSet *gmset = GammaMatrixSet::New(str_gmset_type);

    //- NB. Chiral has not been implemented for SF, yet.
    // Fopr_Clover_SF* fopr_c  = new Fopr_Clover_SF(str_gmset_type);
    Fopr_Clover_SF *fopr_c = new Fopr_Clover_SF();
    fopr_c->set_parameters(*params_clover);

    Fopr_Smeared *fopr_smear = new Fopr_Smeared((Fopr *)fopr_c, dr_smear);
    Fopr         *fopr       = fopr_smear;
    //- NB. U will be converted to Usmear in fopr->set_config
    fopr->set_config(U);

    Solver *solver = Solver::New(str_solver_type, fopr);
    solver->set_parameters(*params_solver);

    Fprop *fprop_lex = new Fprop_Standard_lex(solver);

    Source_Wall_SF *source = new Source_Wall_SF();
    source->set_parameters(*params_source);
    source->set_config(Usmear);


    // ####  Execution main part  ####
    valarray<Field_F> H(Nc * Nd);
    valarray<Field_F> Hpr(Nc * Nd);
    Field_F           xq, b, b2;

    int    Nconv;
    double diff, diff2;

    vout.general(vl, "\n");
    vout.general(vl, "Solving quark propagator:\n");
    vout.general(vl, "  color spin   Nconv      diff           diff2\n");

    for (int icolor = 0; icolor < Nc; ++icolor) {
      for (int ispin = 0; ispin < Nd / 2; ++ispin) {
        source->set_t0(b, icolor, ispin);

        int idx = icolor + Nc * ispin;
        fprop_lex->invert_D(xq, b, Nconv, diff);

        Field_F y(b);
        fopr->set_mode("D");
        y    -= (Field_F)fopr->mult(xq);
        diff2 = y.norm2();
        vout.general(vl, "   %2d   %2d   %6d   %12.4e   %12.4e\n",
                     icolor, ispin, Nconv, diff, diff2);

        H[idx]   = xq;
        xq       = 0.0;
        Hpr[idx] = xq;
      }

      for (int ispin = Nd / 2; ispin < Nd; ++ispin) {
        source->set_tT(b, icolor, ispin);

        int idx = icolor + Nc * ispin;
        fprop_lex->invert_D(xq, b, Nconv, diff);

        Field_F y(b);
        fopr->set_mode("D");
        y    -= (Field_F)fopr->mult(xq);
        diff2 = y.norm2();
        vout.general(vl, "   %2d   %2d   %6d   %12.4e   %12.4e\n",
                     icolor, ispin, Nconv, diff, diff2);

        Hpr[idx] = xq;
        xq       = 0.0;
        H[idx]   = xq;
      }
    }

#if 0
    //For debug
    Index_lex index;
    for (int t = 0; t < 2; ++t) {
      int site = index.site(0, 0, 0, t);
      for (int c1 = 0; c1 < Nc; ++c1) {
        for (int c0 = 0; c0 < Nc; ++c0) {
          for (int s1 = 0; s1 < Nd; ++s1) {
            for (int s0 = 0; s0 < Nd; ++s0) {
              vout.general(vl, "H[%d,%d,%d,%d,%d]=%lf %lf\n",
                           t, c1, c0, s1, s0,
                           H[c0 + Nc * s0].cmp_r(c1, s1, site),
                           H[c0 + Nc * s0].cmp_i(c1, s1, site));
            }
          }
        }
      }
    }
    for (int t = 0; t < 2; ++t) {
      int site = index.site(0, 0, 0, t);
      for (int c1 = 0; c1 < Nc; ++c1) {
        for (int c0 = 0; c0 < Nc; ++c0) {
          for (int s1 = 0; s1 < Nd; ++s1) {
            for (int s0 = 0; s0 < Nd; ++s0) {
              vout.general(vl, "Hpr[%d,%d,%d,%d,%d]=%lf %lf\n",
                           t, c1, c0, s1, s0,
                           Hpr[c0 + Nc * s0].cmp_r(c1, s1, site),
                           Hpr[c0 + Nc * s0].cmp_i(c1, s1, site));
            }
          }
        }
      }
    }
#endif

    //- meson correlators
    vout.general(vl, "\n");
    vout.general(vl, "boundary 2-point correlator with SF BC:\n");

    Corr2pt_Wilson_SF corr(gmset);
    double            result = corr.fAfP(H, Hpr);


    // #####  tidy up  #####
    delete U;
    delete gconf_read;

    delete dr_smear;
    delete fprop_lex;

    delete fopr_c;
    delete fopr_smear;

    delete gmset;
    delete proj;
    delete smear;
    delete solver;
    delete source;

    delete params_test;
    delete params_clover;
    delete params_proj;
    delete params_smear;
    delete params_dr_smear;
    delete params_solver;
    delete params_source;
    delete params_all;


#ifdef USE_TEST
    return Test::verify(expected_result, result);

#else
    return EXIT_SUCCESS;
#endif
  }
} // namespace Test_SF_fAfP
