/*!
        @file    $Id: test_ShiftSolver.cpp #$

        @brief

        @author  Hideo Matsufuru <hideo.matsufuru@kek.jp> (matsufuru)
                 $LastChangedBy: aoym $

        @date    $LastChangedDate: 2013-04-27 12:28:50 #$

        @version $LastChangedRevision: 875 $
*/

#include "parameterManager_YAML.h"
#include "parameters_factory.h"

#include "bridgeIO.h"
using Bridge::vout;

#include "gaugeConfig.h"

#include "fopr_Wilson.h"

#include "index_lex.h"
#include "fprop_Wilson_Shift.h"

#include "source.h"

#ifdef USE_TEST
#include "test.h"
#endif

#ifdef USE_TESTMANAGER_AUTOREGISTER
#include "testManager.h"
#endif

//====================================================================
//! Test of multishift solver.

/*!
                               [28 Dec 2011 H.Matsufuru]
    (Coding history will be recovered from trac.)
    YAML is implemented.       [14 Nov 2012 Y.Namekawa]
    Selectors are implemented. [03 Mar 2013 Y.Namekawa]
    (Selectors are replaced with factories by Aoyama-san)
*/

namespace Test_ShiftSolver {
  //- test-private parameters
  namespace {
    const std::string filename_input  = "test_ShiftSolver.yaml";
    const std::string filename_output = "stdout";

    class Parameters_Test_ShiftSolver : public Parameters {
     public:
      Parameters_Test_ShiftSolver()
      {
        Register_string("gauge_config_type_input", "NULL");
        Register_string("config_filename_input", "NULL");

        Register_string("verbose_level", "NULL");

        Register_double("expected_result", 0.0);
      }
    };
  }

  //- prototype declaration
  int solve(void);

#ifdef USE_TESTMANAGER_AUTOREGISTER
  namespace {
    static const bool is_registered = TestManager::RegisterTest(
      "ShiftSolver",
      solve
      );
  }
#endif

  //====================================================================
  int solve(void)
  {
    // ####  parameter setup  ####
    int Nc   = CommonParameters::Nc();
    int Nd   = CommonParameters::Nd();
    int Ndim = CommonParameters::Ndim();
    int Nvol = CommonParameters::Nvol();

    Parameters *params_test   = new Parameters_Test_ShiftSolver;
    Parameters *params_wilson = ParametersFactory::New("Fopr.Wilson");
    Parameters *params_solver = ParametersFactory::New("Fprop_Wilson_Shift");
    Parameters *params_source = ParametersFactory::New("Source");

    Parameters *params_all = new Parameters;

    params_all->Register_Parameters("Test_ShiftSolver", params_test);
    params_all->Register_Parameters("Fopr_Wilson", params_wilson);
    params_all->Register_Parameters("Fprop_Shift", params_solver);
    params_all->Register_Parameters("Source", params_source);

    ParameterManager_YAML params_manager;
    params_manager.read_params(filename_input, params_all);

    const string str_gconf_read = params_test->get_string("gauge_config_type_input");
    const string readfile       = params_test->get_string("config_filename_input");
    const string str_vlevel     = params_test->get_string("verbose_level");
#ifdef USE_TEST
    const double expected_result = params_test->get_double("expected_result");
#endif

    const string str_gmset_type  = params_wilson->get_string("gamma_matrix_type");
    const int    Nshift          = params_solver->get_int("number_of_shifts");
    const string str_source_type = params_source->get_string("source_type");

    Bridge::VerboseLevel vl = vout.set_verbose_level(str_vlevel);

    //- print input parameters
    vout.general(vl, "  gconf_read  = %s\n", str_gconf_read.c_str());
    vout.general(vl, "  readfile    = %s\n", readfile.c_str());
    vout.general(vl, "  vlevel      = %s\n", str_vlevel.c_str());
    vout.general(vl, "  gmset_type  = %s\n", str_gmset_type.c_str());
    vout.general(vl, "  Nshift      = %d\n", Nshift);
    vout.general(vl, "  source_type = %s\n", str_source_type.c_str());
    vout.general(vl, "\n");

    //- input parameter check
    int err = 0;
    err += ParameterCheck::non_NULL(str_gconf_read);
    err += ParameterCheck::non_NULL(readfile);

    if (err) {
      vout.crucial(vl, "Test_ShiftSolver: Input parameters have not been set.\n");
      abort();
    }


    // ####  object setup  ####
    Field_G     *U          = new Field_G(Nvol, Ndim);
    GaugeConfig *gconf_read = new GaugeConfig(str_gconf_read);
    gconf_read->read_file((Field *)U, readfile);
    // gconf_read->set_cold((Field*)U);

    Fopr_Wilson *fopr_w = new Fopr_Wilson(str_gmset_type);
    fopr_w->set_parameters(*params_wilson);
    fopr_w->set_config(U);

    Source *source = Source::New(str_source_type);
    source->set_parameters(*params_source);

    Index_lex *index = new Index_lex;

    Fprop_Wilson_Shift *fprop_shift = new Fprop_Wilson_Shift(fopr_w, index);
    fprop_shift->set_parameters(*params_solver);


    // ####  Execution main part  ####
    Field_F                b;
    std::valarray<Field_F> xq_shift(Nshift);

    double result;
    {
      int ispin = 0;
      {
        int icolor = 0;
        //  for(int ispin = 0; ispin < Nd; ++ispin){
        //    for(int icolor = 0; icolor < Nc; ++icolor){

        int idx = icolor + Nc * ispin;
        source->set(b, idx);
        result = fprop_shift->calc(&xq_shift, b);
      }
    }


    // ####  tidy up  ####
    delete params_test;
    delete params_wilson;
    delete params_solver;
    delete params_source;
    delete params_all;

    delete U;
    delete gconf_read;

    delete index;    // added by s.motoki[12/06/05].
    delete fprop_shift;
    delete source;
    delete fopr_w;


#ifdef USE_TEST
    return Test::verify(expected_result, result);

#else
    return EXIT_SUCCESS;
#endif
  }
} // namespace Test_ShiftSolver
