/*!
        @file    $Id: test_Spectrum_CRSMatrix_Clover_Lexical.cpp #$

        @brief

        @author  Hideo Matsufuru <hideo.matsufuru@kek.jp> (matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate: 2013-01-22 22:08:29 #$

        @version $LastChangedRevision: 936 $
*/

#include "parameterManager_YAML.h"

#include "bridgeIO.h"
using Bridge::vout;

#include "fopr_CRS.h"
#include "fopr_Clover.h"

#include "source_4spinor_Local.h"

#include "solver_CG.h"

#include "gaugeConfig.h"

#ifdef USE_TEST
#include "test.h"
#endif

#ifdef USE_TESTMANAGER_AUTOREGISTER
#include "testManager.h"
#endif

//====================================================================
//! Test of CRS matrix format.

/*!
    This class generates and tests the CRS matrix format data.
    Matrix data in CRS format are generated for the following
    fermion operators together with sample source and solution
    vectors.

    The egenrated data can be tested using the function
    CRSsolver() whether really solved result satisfy the linear
    equation.
                                [24 Dec 2011 H.Matsufuru]
    (Coding history will be recovered from trac.)
    YAML is implemented.        [14 Nov 2012 Y.Namekawa]
 */

namespace Test_Spectrum_CRSMatrix {
  //- test-private parameters
  namespace {
    const std::string filename_input  = "test_Spectrum_CRSMatrix_Clover_Lexical.yaml";
    const std::string filename_output = "stdout";

    class Parameters_Test_Spectrum_CRSMatrix : public Parameters {
     public:
      Parameters_Test_Spectrum_CRSMatrix()
      {
        Register_string("gauge_config_type_input", "NULL");
        Register_string("config_filename_input", "NULL");

        Register_string("matrix_output", "NULL");
        Register_string("source_output", "NULL");
        Register_string("solution_output", "NULL");

        Register_string("verbose_level", "NULL");

        Register_double("expected_result", 0.0);
      }
    };
  }

  //- prototype declaration
  int clover_lex(void);

  int CRSsolver(
    const string& solution,
    const string& matrix,
    const string& source,
    double& result       /* return value */
    );


#ifdef USE_TESTMANAGER_AUTOREGISTER
#ifdef USE_MPI
  // this test runs only in single-node environment.
#else
  //- NB. CRS test is skipped, because it is time-consuming.
  // namespace {
  //   static const bool is_registered = TestManager::RegisterTest(
  //     "Spectrum.CRSMatrix.Clover_Lexical",
  //     clover_lex
  //     );
  // }
#endif
#endif

  //====================================================================
  int clover_lex(void)
  {
    // ####  parameter setup  ####
    int Nc   = CommonParameters::Nc();
    int Nd   = CommonParameters::Nd();
    int Ndim = CommonParameters::Ndim();
    int Nvol = CommonParameters::Nvol();

    Parameters_Test_Spectrum_CRSMatrix params_test;

    Parameters params_all;

    params_all.Register_Parameters("Test_Spectrum_CRSMatrix", &params_test);

    ParameterManager_YAML params_manager;
    params_manager.read_params(filename_input, &params_all);

    const string str_gconf_read = params_test.get_string("gauge_config_type_input");
    const string readfile       = params_test.get_string("config_filename_input");
    const string matrix_file    = params_test.get_string("matrix_output");
    const string source_file    = params_test.get_string("source_output");
    const string solution_file  = params_test.get_string("solution_output");
    const string str_vlevel     = params_test.get_string("verbose_level");
#ifdef USE_TEST
    const double expected_result = params_test.get_double("expected_result");
#endif

    Bridge::VerboseLevel vl = vout.set_verbose_level(str_vlevel);

    //- print input parameters
    vout.general(vl, "  gconf_read      = %s\n", str_gconf_read.c_str());
    vout.general(vl, "  readfile        = %s\n", readfile.c_str());
    vout.general(vl, "  matrix_output   = %s\n", matrix_file.c_str());
    vout.general(vl, "  source_output   = %s\n", source_file.c_str());
    vout.general(vl, "  solution_output = %s\n", solution_file.c_str());
    vout.general(vl, "  vlevel          = %s\n", str_vlevel.c_str());
    vout.general(vl, "\n");

    //- input parameter check
    int err = 0;
    err += ParameterCheck::non_NULL(str_gconf_read);
    err += ParameterCheck::non_NULL(readfile);
    err += ParameterCheck::non_NULL(matrix_file);
    err += ParameterCheck::non_NULL(source_file);
    err += ParameterCheck::non_NULL(solution_file);

    if (err) {
      vout.crucial(vl, "Test_Spectrum_CRSMatrix: Input parameters have not been set.\n");
      abort();
    }


    // ####  Set up a gauge configuration  ####
    Field_G     *U          = new Field_G(Nvol, Ndim);
    GaugeConfig *gconf_read = new GaugeConfig(str_gconf_read);
    gconf_read->read_file((Field *)U, readfile);
    // gconf_read->set_cold((Field*)U);


    // ####  object setup  #####
    Fopr_Clover   *fopr_c = new Fopr_Clover();
    double        kappa   = 0.12;
    double        cSW     = 1.0;
    valarray<int> boundary(Ndim);
    boundary[0] = -1;
    boundary[1] = -1;
    boundary[2] = -1;
    boundary[3] = -1;
    fopr_c->set_parameters(kappa, cSW, boundary);
    fopr_c->set_config(U);
    fopr_c->set_mode("D");

    Fopr_CRS *fopr_crs = new Fopr_CRS(fopr_c);
    fopr_crs->write_matrix(matrix_file);

    int    Niter     = 5000;
    double Stop_cond = 1.0e-28;
    Solver *solver   = new Solver_CG((Fopr *)fopr_c);
    solver->set_parameters(Niter, Stop_cond);

    valarray<int> source_position(Ndim);
    source_position[0] = 0;
    source_position[1] = 0;
    source_position[2] = 0;
    source_position[3] = 0;

    Source_4spinor_Local *source = new Source_4spinor_Local();
    source->set_parameters(source_position);


    // ####  Execution main part  ####
    valarray<Field_F> sq(Nc * Nd);
    Field_F           xq, b, b2;

    {
      int ispin = 0;
      {
        int icolor = 0;
        //for(int ispin = 0; ispin < Nd; ++ispin){
        // for(int icolor = 0; icolor < Nc; ++icolor){

        source->set(b, icolor, ispin);

        b.write_text("source4x8_clover.crs");

        fopr_c->set_mode("D");
        b2 = (Field_F)fopr_c->mult_dag(b);

        int    Nconv;
        double diff_CG;
        fopr_c->set_mode("DdagD");
        solver->solve(xq, b2, Nconv, diff_CG);
        vout.general(vl, "  ispin = %2d  icolor = %2d  Nconv = %4d  diff = %12.6e\n",
                     ispin, icolor, Nconv, diff_CG);

        xq.write_text("solution4x8_clover.crs");

        Field y(b);
        fopr_c->set_mode("D");
        y -= fopr_c->mult(xq);
        double yy = y.norm2();
        vout.general(vl, "    standard norm2 = %.8e\n", yy);

        sq[icolor + Nc * ispin] = xq;
      }
    }

#if 0
    // ### Solver for CRS version ###

    Solver *solver_crs = new Solver_CG((Fopr *)fopr_crs2);
    solver_crs->set_parameters(Niter, Stop_cond);

    double yy = 0.0L;
    {
      int ispin = 0;
      {
        int icolor = 0;
        //for(int ispin = 0; ispin < Nd; ++ispin){
        // for(int icolor = 0; icolor < Nc; ++icolor){

        source->set(b, icolor, ispin);

        fopr_crs2->set_mode("Ddag");
        b2 = (Field_F)fopr_crs2->mult((Field)b);

        int    Nconv;
        double diff_CG;
        fopr_crs2->set_mode("DdagD");
        solver_crs->solve(xq, b2, Nconv, diff_CG);
        vout.general(vl, "  ispin = %2d  icolor = %2d  Nconv = %4d  diff = %12.6e\n",
                     ispin, icolor, Nconv, diff_CG);

        Field_F y(b);
        fopr_crs2->set_mode("D");
        y -= (Field_F)fopr_crs2->mult((Field)xq);
        yy = y.norm2();
        vout.general(vl, "    standard norm2 = %.8e\n", yy);

        sq[icolor + Nc * ispin] = xq;
      }
    }
#endif


    double result;
    CRSsolver(solution_file, matrix_file, source_file, result);


    // ####  tidy up  ####
    delete U;
    delete gconf_read;

    delete source;
    delete solver;

    delete fopr_crs;
    delete fopr_c;

#if 0
    delete solver_crs;
    delete fopr_crs2;
#endif


#ifdef USE_TEST
    return Test::verify(expected_result, result);

#else
    return EXIT_SUCCESS;
#endif
  }
} // namespace Test_Spectrum_CRSMatrix
