/*!
        @file    $Id:: test_Spectrum_Clover_2ptFunction.cpp #$

        @brief

        @author  Hideo Matsufuru <hideo.matsufuru@kek.jp> (matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-19 14:15:23 #$

        @version $LastChangedRevision: 936 $
*/

#include <sstream>
#include "parameterManager_YAML.h"
#include "parameters_factory.h"

#include "bridgeIO.h"
using Bridge::vout;

#include "gaugeConfig.h"
#include "staples.h"

#include "randomNumbers_Mseries.h"

#include "fopr_Clover.h"
#include "fprop_Standard_lex.h"

#include "director_Smear.h"
#include "fopr_Smeared.h"

#include "gammaMatrixSet.h"
#include "gaugeFixing.h"
#include "projection.h"
#include "smear.h"
#include "solver.h"
#include "source.h"

#include "corr2pt_4spinor.h"

#ifdef USE_TEST
#include "test.h"
#endif

#ifdef USE_TESTMANAGER_AUTOREGISTER
#include "testManager.h"
#endif

//====================================================================
//! Test of spectroscopy with clover fermion.

/*!
    This test class obtains the quark propagator for clover fermion
    and calculates typical hadron correlators.
    The quantum numbers of hadrons are specified with gamma matrices
    (GammaMatrix class instance) whose set is defined in a subclass
    of GammaMatrixSet class.
                                             [12 Apr 2012 H.Matsufuru]
    (Coding history will be recovered from trac.)
    Selectors are implemented.               [02 Feb 2013 Y.Namekawa]
    Smearing is implemented.                 [03 Mar 2013 Y.Namekawa]
    (Selectors are replaced with factories by Aoyama-san)
    Number_of_valence_quarks is implemented. [15 May 2013 Y.Namekawa]
*/

namespace Test_Spectrum_Clover {
  //- test-private parameters
  namespace {
    const std::string filename_input  = "test_Spectrum_Clover_Hadron2ptFunction.yaml";
    const std::string filename_output = "stdout";

    class Parameters_Test_Spectrum_Clover : public Parameters {
     public:
      Parameters_Test_Spectrum_Clover()
      {
        Register_string("gauge_config_type_input", "NULL");
        Register_string("config_filename_input", "NULL");

        Register_int("number_of_valence_quarks", 0);

        Register_string("verbose_level", "NULL");

        Register_double("expected_result", 0.0);
      }
    };

    class Parameters_Quark : virtual public Parameters {
     public:
      Parameters_Quark()
      {
        Register_Parameters("Fopr_Clover", ParametersFactory::New("Fopr.Clover"));
        Register_Parameters("Source", ParametersFactory::New("Source"));
      }
    };
  }

  //- prototype declaration
  int hadron_2ptFunction(void);

#ifdef USE_TESTMANAGER_AUTOREGISTER
  namespace {
    static const bool is_registered = TestManager::RegisterTest(
      "Spectrum.Clover.Hadron2ptFunction",
      hadron_2ptFunction
      );
  }
#endif

  //====================================================================
  int hadron_2ptFunction(void)
  {
    // ####  parameter setup  ####
    int Nc   = CommonParameters::Nc();
    int Nd   = CommonParameters::Nd();
    int Ndim = CommonParameters::Ndim();
    int Nvol = CommonParameters::Nvol();

    ParameterManager_YAML params_manager;

    Parameters *params_test = new Parameters_Test_Spectrum_Clover;

    Parameters *params_pre = new Parameters;

    params_pre->Register_Parameters("Test_Spectrum_Clover", params_test);

    params_manager.read_params(filename_input, params_pre);


    const int                 N_quark = params_test->get_int("number_of_valence_quarks");
    std::vector<Parameters *> params_quark(N_quark);

    for (int iq = 0; iq < N_quark; ++iq) {
      params_quark[iq] = new Parameters_Quark;
    }

    Parameters *params_gfix     = ParametersFactory::New("GaugeFixing");
    Parameters *params_proj     = ParametersFactory::New("Projection");
    Parameters *params_smear    = ParametersFactory::New("Smear");
    Parameters *params_dr_smear = ParametersFactory::New("Director_Smear");
    Parameters *params_solver   = ParametersFactory::New("Solver");

    Parameters *params_all = new Parameters;


    std::stringstream oss;

    for (int iq = 0; iq < N_quark; ++iq) {
      oss.str("");
      oss << "Quark_" << iq + 1;
      string str = oss.str();
      params_all->Register_Parameters(str.c_str(), params_quark[iq]);
    }

    params_all->Register_Parameters("GaugeFixing", params_gfix);
    params_all->Register_Parameters("Projection", params_proj);
    params_all->Register_Parameters("Smear", params_smear);
    params_all->Register_Parameters("Director_Smear", params_dr_smear);
    params_all->Register_Parameters("Solver", params_solver);

    params_manager.read_params(filename_input, params_all);

    const string str_gconf_read = params_test->get_string("gauge_config_type_input");
    const string readfile       = params_test->get_string("config_filename_input");
    const string str_vlevel     = params_test->get_string("verbose_level");
#ifdef USE_TEST
    const double expected_result = params_test->get_double("expected_result");
#endif

    const string str_gfix_type = params_gfix->get_string("gauge_fixing_type");

    std::vector<Parameters *> params_clover(N_quark);
    std::vector<Parameters *> params_source(N_quark);

    for (int iq = 0; iq < N_quark; ++iq) {
      params_clover[iq] = params_quark[iq]->get_Parameters("Fopr_Clover");
      params_source[iq] = params_quark[iq]->get_Parameters("Source");
    }

    const string str_proj_type   = params_proj->get_string("projection_type");
    const string str_smear_type  = params_smear->get_string("smear_type");
    const string str_solver_type = params_solver->get_string("solver_type");

    Bridge::VerboseLevel vl = vout.set_verbose_level(str_vlevel);

    //- print input parameters
    vout.general(vl, "  gconf_read  = %s\n", str_gconf_read.c_str());
    vout.general(vl, "  readfile    = %s\n", readfile.c_str());
    vout.general(vl, "  vlevel      = %s\n", str_vlevel.c_str());
    vout.general(vl, "  gfix_type   = %s\n", str_gfix_type.c_str());
    vout.general(vl, "  proj_type   = %s\n", str_proj_type.c_str());
    vout.general(vl, "  smear_type  = %s\n", str_smear_type.c_str());
    vout.general(vl, "  solver_type = %s\n", str_solver_type.c_str());

    // NB. all str_gmset_type are supposed to be the same.
    string str_gmset_type = params_clover[0]->get_string("gamma_matrix_type");
    vout.general(vl, "  gmset_type  = %s\n", str_gmset_type.c_str());

    std::vector<std::string> str_source_type(N_quark);

    for (int iq = 0; iq < N_quark; ++iq) {
      vout.general(vl, "  Quark_%d:\n", iq + 1);

      str_source_type[iq] = params_source[iq]->get_string("source_type");
      vout.general(vl, "    source_type = %s\n", str_source_type[iq].c_str());
    }
    vout.general(vl, "\n");

    //- input parameter check
    int err = 0;
    err += ParameterCheck::non_NULL(str_gconf_read);
    err += ParameterCheck::non_NULL(readfile);

    if (err) {
      vout.crucial(vl, "Test_Spectrum_Clover: Input parameters have not been set.\n");
      abort();
    }


    // ####  Set up a gauge configuration  ####
    Field_G     *U          = new Field_G(Nvol, Ndim);
    GaugeConfig *gconf_read = new GaugeConfig(str_gconf_read);
    gconf_read->read_file((Field *)U, readfile);
    // gconf_read->set_cold((Field*)U);

    Projection *proj  = Projection::New(str_proj_type);
    Smear      *smear = Smear::New(str_smear_type, proj);
    smear->set_parameters(*params_smear);

    Director_Smear *dr_smear = new Director_Smear((Smear *)smear);
    dr_smear->set_parameters(*params_dr_smear);
    dr_smear->set_config(U);

    int     Nsmear  = dr_smear->get_Nsmear();
    Field_G *Usmear = (Field_G *)dr_smear->getptr_smearedConfig(Nsmear);


    // ####  Gauge fixing  ####
    Staples *staple = new Staples;
    Field_G *Ufix   = new Field_G(Nvol, Ndim);

    int           ndelay = 1000;
    RandomNumbers *rand  = new RandomNumbers_Mseries(ndelay);

    GaugeFixing *gfix = GaugeFixing::New(str_gfix_type, rand);
    gfix->set_parameters(*params_gfix);

    double plaq = staple->plaquette(*Usmear);
    vout.general(vl, "plaq(original) = %18.14f\n", plaq);

    gfix->fix(*Ufix, *Usmear);

    double plaq2 = staple->plaquette(*Ufix);
    vout.general(vl, "plaq(fixed)    = %18.14f\n", plaq2);
    vout.general(vl, "plaq(diff)     = %18.10e\n", plaq - plaq2);


    // ####  object setup  #####
    GammaMatrixSet *gmset = GammaMatrixSet::New(str_gmset_type);

    std::vector<Fopr_Clover *> fopr_c(N_quark);
    std::vector<Solver *>      solver(N_quark);
    std::vector<Fprop *>       fprop_lex(N_quark);
    std::vector<Source *>      source(N_quark);

    for (int iq = 0; iq < N_quark; ++iq) {
      fopr_c[iq] = new Fopr_Clover(str_gmset_type);
      fopr_c[iq]->set_parameters(*params_clover[iq]);
      fopr_c[iq]->set_config(Ufix);

      solver[iq] = Solver::New(str_solver_type, fopr_c[iq]);
      solver[iq]->set_parameters(*params_solver);

      fprop_lex[iq] = new Fprop_Standard_lex(solver[iq]);

      source[iq] = Source::New(str_source_type[iq]);
      source[iq]->set_parameters(*params_source[iq]);
    }
    vout.general(vl, "\n");


    // ####  Execution main part  ####
    typedef std::valarray<Field_F>   PropagatorSet;

    std::vector<PropagatorSet> sq(N_quark);
    for (int iq = 0; iq < N_quark; ++iq) {
      sq[iq].resize(Nc * Nd);

      for (int i = 0; i < Nc * Nd; ++i) {
        sq[iq][i] = 0.0;
      }
    }

    Field_F b;
    b = 0.0;

    int    Nconv;
    double diff, diff2;

    for (int iq = 0; iq < N_quark; ++iq) {
      vout.general(vl, "Solving quark propagator, flavor = %d:\n", iq + 1);
      vout.general(vl, "  color spin   Nconv      diff           diff2\n");

      for (int ispin = 0; ispin < Nd; ++ispin) {
        for (int icolor = 0; icolor < Nc; ++icolor) {
          int idx = icolor + Nc * ispin;
          source[iq]->set(b, idx);

          fprop_lex[iq]->invert_D(sq[iq][idx], b, Nconv, diff);

          Field_F y(b);
          fopr_c[iq]->set_mode("D");
          y    -= (Field_F)fopr_c[iq]->mult(sq[iq][idx]);
          diff2 = y.norm2();

          vout.general(vl, "   %2d   %2d   %6d   %12.4e   %12.4e\n",
                       icolor, ispin, Nconv, diff, diff2);
        }
      }

      vout.general(vl, "\n");
    }


    //- meson correlators
    std::ofstream log_file;
    if (filename_output != "stdout") {
      log_file.open(filename_output.c_str());
      vout.init(log_file);
    }

    vout.general(vl, "2-point correlator:\n");
    Corr2pt_4spinor  corr(gmset);
    valarray<double> result(N_quark);

    //- case(iq_1 == iq_2)
    for (int iq = 0; iq < N_quark; ++iq) {
      vout.general(vl, "Flavor combination = %d, %d\n", iq + 1, iq + 1);
      result[iq] = corr.meson_all(sq[iq], sq[iq]);
      vout.general(vl, "\n");
    }


    //- case(iq_1 < iq_2)
    for (int iq = 0; iq < N_quark; ++iq) {
      for (int jq = iq + 1; jq < N_quark; ++jq) {
        vout.general(vl, "Flavor combination = %d, %d\n", iq + 1, jq + 1);
        double result_2 = corr.meson_all(sq[iq], sq[jq]);
        vout.general(vl, "\n");
      }
    }

    if (filename_output != "stdout") {
      log_file.close();
      vout.init(std::cout);
    }


    // ####  tydy up  ####
    delete gconf_read;
    delete U;
    delete Ufix;

    delete dr_smear;

    delete rand;
    delete staple;

    for (int iq = 0; iq < N_quark; ++iq) {
      delete fopr_c[iq];
      delete solver[iq];
      delete fprop_lex[iq];
      delete source[iq];
    }

    delete gfix;
    delete gmset;
    delete proj;
    delete smear;

    delete params_pre;
    delete params_test;
    delete params_gfix;
    delete params_proj;
    delete params_smear;
    delete params_dr_smear;
    delete params_solver;

    for (int iq = 0; iq < N_quark; ++iq) {
      delete params_clover[iq];
      delete params_source[iq];
      delete params_quark[iq];
    }

    delete params_all;


#ifdef USE_TEST
    return Test::verify(expected_result, result[0]);

#else
    return EXIT_SUCCESS;
#endif
  }
} // namespace Test_Spectrum_Clover
