/*!
        @file    $Id: test_Spectrum_Wilson_2ptFunction_eo.cpp #$

        @brief

        @author  UEDA, Satoru <sueda@post.kek.jp> (sueda)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-19 14:15:23 #$

        @version $LastChangedRevision: 936 $
*/

#include "parameterManager_YAML.h"
#include "parameters_factory.h"

#include "bridgeIO.h"
using Bridge::vout;

#include "gaugeConfig.h"
#include "staples.h"

#include "randomNumbers_Mseries.h"

#include "fopr_Wilson.h"
#include "fopr_Wilson_eo.h"
#include "fprop_Standard_eo.h"

#include "gaugeFixing.h"
#include "gammaMatrixSet.h"
#include "solver.h"
#include "source.h"

#include "corr2pt_4spinor.h"

#ifdef USE_TEST
#include "test.h"
#endif

#ifdef USE_TESTMANAGER_AUTOREGISTER
#include "testManager.h"
#endif

//====================================================================
//! Test of Wilson fermion solver with even-odd preconditioning.

/*!
                                          [28 Dec 2011 H.Matsufuru]
    (Coding history will be recovered from trac.)
    YAML is implemented.                  [14 Nov 2012 Y.Namekawa]
    Selectors are implemented.            [03 Mar 2013 Y.Namekawa]
    (Selectors are replaced with factories by Aoyama-san)
 */

namespace Test_Spectrum_Wilson {
  //- test-private parameters
  namespace {
    const std::string filename_input  = "test_Spectrum_Wilson_Hadron2ptFunction.yaml";
    const std::string filename_output = "stdout";

    class Parameters_Test_Spectrum_Wilson : public Parameters {
     public:
      Parameters_Test_Spectrum_Wilson()
      {
        Register_string("gauge_config_type_input", "NULL");
        Register_string("config_filename_input", "NULL");

        Register_string("verbose_level", "NULL");

        Register_double("expected_result", 0.0);
      }
    };
  }

  //- prototype declaration
  int hadron_2ptFunction_eo(void);

#ifdef USE_TESTMANAGER_AUTOREGISTER
  namespace {
    static const bool is_registered = TestManager::RegisterTest(
      "Spectrum.Wilson.Hadron2ptFunction_eo",
      hadron_2ptFunction_eo
      );
  }
#endif

  //====================================================================
  int hadron_2ptFunction_eo(void)
  {
    // ####  parameter setup  ####
    int Nc   = CommonParameters::Nc();
    int Nd   = CommonParameters::Nd();
    int Nvol = CommonParameters::Nvol();
    int Ndim = CommonParameters::Ndim();

    Parameters *params_test   = new Parameters_Test_Spectrum_Wilson;
    Parameters *params_gfix   = ParametersFactory::New("GaugeFixing");
    Parameters *params_wilson = ParametersFactory::New("Fopr.Wilson");
    Parameters *params_solver = ParametersFactory::New("Solver");
    Parameters *params_source = ParametersFactory::New("Source");

    Parameters *params_all = new Parameters;

    params_all->Register_Parameters("Test_Spectrum_Wilson", params_test);
    params_all->Register_Parameters("GaugeFixing", params_gfix);
    params_all->Register_Parameters("Fopr_Wilson", params_wilson);
    params_all->Register_Parameters("Solver", params_solver);
    params_all->Register_Parameters("Source", params_source);

    ParameterManager_YAML params_manager;
    params_manager.read_params(filename_input, params_all);

    const string str_gconf_read = params_test->get_string("gauge_config_type_input");
    const string readfile       = params_test->get_string("config_filename_input");
    const string str_vlevel     = params_test->get_string("verbose_level");
#ifdef USE_TEST
    const double expected_result = params_test->get_double("expected_result");
#endif

    const string str_gfix_type   = params_gfix->get_string("gauge_fixing_type");
    const string str_gmset_type  = params_wilson->get_string("gamma_matrix_type");
    const string str_solver_type = params_solver->get_string("solver_type");
    const string str_source_type = params_source->get_string("source_type");

    Bridge::VerboseLevel vl = vout.set_verbose_level(str_vlevel);

    //- print input parameters
    vout.general(vl, "  gconf_read  = %s\n", str_gconf_read.c_str());
    vout.general(vl, "  readfile    = %s\n", readfile.c_str());
    vout.general(vl, "  vlevel      = %s\n", str_vlevel.c_str());
    vout.general(vl, "  gfix_type   = %s\n", str_gfix_type.c_str());
    vout.general(vl, "  gmset_type  = %s\n", str_gmset_type.c_str());
    vout.general(vl, "  solver_type = %s\n", str_solver_type.c_str());
    vout.general(vl, "  source_type = %s\n", str_source_type.c_str());

    //- input parameter check
    int err = 0;
    err += ParameterCheck::non_NULL(str_gconf_read);
    err += ParameterCheck::non_NULL(readfile);

    if (err) {
      vout.crucial(vl, "Test_Spectrum_Wilson: Input parameters have not been set.\n");
      abort();
    }


    // ####  Set up a gauge configuration  ####
    Field_G     *U          = new Field_G(Nvol, Ndim);
    GaugeConfig *gconf_read = new GaugeConfig(str_gconf_read);
    gconf_read->read_file((Field *)U, readfile);
    // gconf_read->set_cold((Field*)U);


    // ####  Gauge fixing  ####
    Staples *staple = new Staples;
    Field_G *Ufix   = new Field_G(Nvol, Ndim);

    int           ndelay = 1000;
    RandomNumbers *rand  = new RandomNumbers_Mseries(ndelay);

    GaugeFixing *gfix = GaugeFixing::New(str_gfix_type, rand);
    gfix->set_parameters(*params_gfix);

    double plaq = staple->plaquette(*U);
    vout.general(vl, "plaq(original) = %18.14f\n", plaq);

    gfix->fix(*Ufix, *U);

    double plaq2 = staple->plaquette(*Ufix);
    vout.general(vl, "plaq(fixed)    = %18.14f\n", plaq2);
    vout.general(vl, "plaq(diff)     = %18.10e\n", plaq - plaq2);


    // ####  object setup  #####
    GammaMatrixSet *gmset = GammaMatrixSet::New(str_gmset_type);

    // NB. Fopr_Wilson is used only for check of diff2 below.
    Fopr_Wilson *fopr_w = new Fopr_Wilson(str_gmset_type);
    fopr_w->set_parameters(*params_wilson);
    fopr_w->set_config(Ufix);

    Fopr_Wilson_eo *fopr_w_eo = new Fopr_Wilson_eo(str_gmset_type);
    fopr_w_eo->set_parameters(*params_wilson);
    fopr_w_eo->set_config(Ufix);

    Solver *solver = Solver::New(str_solver_type, fopr_w_eo);
    solver->set_parameters(*params_solver);

    Fprop *fprop_eo = new Fprop_Standard_eo(solver);

    Source *source = Source::New(str_source_type);
    source->set_parameters(*params_source);


    // ####  Execution main part  ####
    std::valarray<Field_F> sq(Nc * Nd);
    for (int i = 0; i < Nc * Nd; ++i) {
      sq[i] = 0.0;
    }

    Field_F b;
    b = 0.0;

    int    Nconv;
    double diff;

    vout.general(vl, "\n");
    vout.general(vl, "Solving quark propagator:\n");
    vout.general(vl, "  color spin   Nconv      diff           diff2\n");

    for (int ispin = 0; ispin < Nd; ++ispin) {
      for (int icolor = 0; icolor < Nc; ++icolor) {
        int idx = icolor + Nc * ispin;
        source->set(b, idx);

        fprop_eo->invert_D(sq[idx], b, Nconv, diff);

        Field_F y(b);
        fopr_w->set_mode("D");
        y -= (Field_F)fopr_w->mult(sq[idx]);
        double diff2 = y.norm();

        vout.general(vl, "   %2d   %2d   %6d   %12.4e   %12.4e\n",
                     icolor, ispin, Nconv, diff, diff2);
      }
    }

    vout.general(vl, "\n");
    vout.general(vl, "2-point correlator:\n");

    Corr2pt_4spinor corr(gmset);
    double          result = corr.meson_all(sq, sq);


    // ####  tidy up  ####
    delete gconf_read;
    delete U;
    delete Ufix;

    delete rand;
    delete staple;

    delete fopr_w;
    delete fopr_w_eo;
    delete fprop_eo;

    delete gfix;
    delete gmset;
    delete solver;
    delete source;

    delete params_test;
    delete params_gfix;
    delete params_wilson;
    delete params_solver;
    delete params_source;
    delete params_all;


#ifdef USE_TEST
    return Test::verify(expected_result, result);

#else
    return EXIT_SUCCESS;
#endif
  }
} // namespace Test_Spectrum_Wilson
