/*!
        @file    $Id: test_WilsonLoop.cpp #$

        @brief

        @author  Hideo Matsufuru <hideo.matsufuru@kek.jp> (matsufuru)
                 $LastChangedBy: aoym $

        @date    $LastChangedDate: 2013-01-22 13:51:53 #$

        @version $LastChangedRevision: 875 $
*/

#include "parameterManager_YAML.h"
#include "parameters_factory.h"

#include "bridgeIO.h"
using Bridge::vout;

#include "gaugeConfig.h"

#include "wilsonLoop.h"

#include "projection_Maximum_SU_N.h"
#include "smear_APE_spatial.h"

#ifdef USE_TEST
#include "test.h"
#endif

#ifdef USE_TESTMANAGER_AUTOREGISTER
#include "testManager.h"
#endif

//====================================================================
//! Test of Wilson loop measurement.

/*!
                          [27 May 2012 H.Matsufuru]
    (Coding history will be recovered from trac.)
    YAML is implemented.  [14 Nov 2012 Y.Namekawa]
 */

namespace Test_WilsonLoop {
  //- test-private parameters
  namespace {
    const std::string filename_input  = "test_WilsonLoop.yaml";
    const std::string filename_output = "stdout";

    class Parameters_Test_WilsonLoop : public Parameters {
     public:
      Parameters_Test_WilsonLoop()
      {
        Register_string("gauge_config_type_input", "NULL");
        Register_string("config_filename_input", "NULL");

        Register_int("number_of_max_smearing", 0);
        Register_int("number_of_smearing_step", 0);

        Register_string("verbose_level", "NULL");

        Register_double("expected_result", 0.0);
      }
    };
  }

  //- prototype declaration
  int wilsonloop(void);

#ifdef USE_TESTMANAGER_AUTOREGISTER
  namespace {
    static const bool is_registered = TestManager::RegisterTest(
      "WilsonLoop",
      wilsonloop
      );
  }
#endif

  //====================================================================
  int wilsonloop(void)
  {
    // ####  parameter setup  ####
    int Nvol = CommonParameters::Nvol();
    int Ndim = CommonParameters::Ndim();

    Parameters *params_test       = new Parameters_Test_WilsonLoop;
    Parameters *params_wilsonloop = ParametersFactory::New("WilsonLoop");
    Parameters *params_smear      = ParametersFactory::New("Smear.APE_Spatial");

    Parameters *params_all = new Parameters;

    params_all->Register_Parameters("Test_WilsonLoop", params_test);
    params_all->Register_Parameters("WilsonLoop", params_wilsonloop);
    params_all->Register_Parameters("Smear_APE_spatial", params_smear);

    ParameterManager_YAML params_manager;
    params_manager.read_params(filename_input, params_all);

    const string str_gconf_read = params_test->get_string("gauge_config_type_input");
    const string readfile       = params_test->get_string("config_filename_input");
    const int    Nsmear         = params_test->get_int("number_of_max_smearing");
    const int    Nmeas          = params_test->get_int("number_of_smearing_step");
    const string str_vlevel     = params_test->get_string("verbose_level");
#ifdef USE_TEST
    const double expected_result = params_test->get_double("expected_result");
#endif

    Bridge::VerboseLevel vl = vout.set_verbose_level(str_vlevel);

    //- print input parameters
    vout.general(vl, "  gconf_read = %s\n", str_gconf_read.c_str());
    vout.general(vl, "  readfile   = %s\n", readfile.c_str());
    vout.general(vl, "  Nsmear     = %d\n", Nsmear);
    vout.general(vl, "  Nmeas      = %d\n", Nmeas);
    vout.general(vl, "  vlevel     = %s\n", str_vlevel.c_str());
    vout.general(vl, "\n");

    //- input parameter check
    int err = 0;
    err += ParameterCheck::non_NULL(str_gconf_read);
    err += ParameterCheck::non_NULL(readfile);

    if (err) {
      vout.crucial(vl, "Test_WilsonLoop: Input parameters have not been set.\n");
      abort();
    }


    // #### object setup #####
    Field_G     *U          = new Field_G(Nvol, Ndim);
    GaugeConfig *gconf_read = new GaugeConfig(str_gconf_read);
    gconf_read->read_file((Field *)U, readfile);
    // gconf_read->set_cold((Field*)U);

    WilsonLoop wilsonloop;
    wilsonloop.set_parameters(*params_wilsonloop);

    Projection        *proj  = new Projection_Maximum_SU_N();
    Smear_APE_spatial *smear = new Smear_APE_spatial(proj);
    smear->set_parameters(*params_smear);


    // ####  Execution main part  ####
    Field_G Uorg(Nvol, Ndim), Usmear(Nvol, Ndim);
    Uorg   = *U;
    Usmear = *U;

    double result = 0.0;
    for (int i_smear = 0; i_smear <= Nsmear; ++i_smear) {
      vout.general(vl, "i_smear = %d\n", i_smear);

      if (i_smear > 0) smear->smear(Usmear, Uorg);

      if ((i_smear % Nmeas) == 0) {
        result = wilsonloop.measure(Usmear);
      }

      Uorg = Usmear;
    }


    // ####  tydy up  ####
    delete params_test;
    delete params_wilsonloop;
    delete params_smear;
    delete params_all;

    delete U;
    delete gconf_read;
    delete proj;
    delete smear;


#ifdef USE_TEST
    return Test::verify(expected_result, result);

#else
    return EXIT_SUCCESS;
#endif
  }
} // namespace Test_WilsonLoop
