/*!
        @file    $Id:: gammaMatrixSet#$

        @brief

        @author  Hideo Matsufuru hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#ifndef GAMMAMATRIXSET_INCLUDED
#define GAMMAMATRIXSET_INCLUDED

#include <valarray>
#include <cassert>

#include "commonParameters.h"
#include "gammaMatrix.h"

#ifdef USE_FACTORY
#include "factory.h"
#endif

//! Set of Gamma Matrices: basis class.

/*!
   This class difines a set of gamma matrices.
   Present implementation is applicable only to Ndim=4.
   Just possible specied of gamma matrices are enumerated in
   this class, and practical form is given in subclass by
   implementing virtual function init_GM().
                                        [4 Feb 2012 H.Matsufuru]
 */

class GammaMatrixSet
{
 protected:
  int m_Nspecies;
  std::valarray<GammaMatrix> m_gm;

  Bridge::VerboseLevel m_vl;

 public:
  enum GMspecies
  {
    UNITY, GAMMA1, GAMMA2, GAMMA3, GAMMA4,
    GAMMA5, GAMMA51, GAMMA52, GAMMA53, GAMMA54,
    SIGMA12, SIGMA23, SIGMA31,
    SIGMA41, SIGMA42, SIGMA43, CHARGECONJG
  };

  GammaMatrixSet()
    : m_vl(CommonParameters::Vlevel())
  {
    int Nd = CommonParameters::Nd();

    assert(Nd == 4);
    m_Nspecies = Nd * Nd + 1;    // must be 17.
    m_gm.resize(m_Nspecies);
  }

  virtual ~GammaMatrixSet() {}

  virtual void init_GM() = 0;

  GammaMatrix get_GM(GMspecies spec)
  {
    assert(spec < m_Nspecies);
    return m_gm[spec];
  }

  virtual void print() = 0;

#ifdef USE_FACTORY
 public:
  typedef GammaMatrixSet *(*ProductCreator)();
  typedef FactoryTemplate<GammaMatrixSet, ProductCreator>   Factory;

  static GammaMatrixSet *New(const IdentifierType& subtype)
  {
    ProductCreator p = Factory::Find(subtype);

    return p ? (*p)() : 0;
  }
#endif
};
#endif
