/*!
        @file    $Id:: randomNumbers.cpp #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-19 14:15:23 #$

        @version $LastChangedRevision: 936 $
*/


#include "randomNumbers.h"

#include <iostream>
#include <fstream>
#include <cassert>

#include "communicator.h"
#include "index_lex.h"
#include "index_eo.h"

using Bridge::vout;

namespace {
  const double sq2r = 1.0 / sqrt(2.0);
//  const double pi   = 3.141592653589793;
  const double pi  = 4.0 * atan(1.0);
  const double pi2 = 2.0 * pi;
}

//====================================================================
void RandomNumbers::gauss(double& rn1, double& rn2)
{
  //   Two Gaussian random number with deviation 1/\sqrt(2).
  double r1 = get();
  double r2 = get();

  //    double slg1 = sqrt( -log(r1)*2.0 )*sq2r;
  double slg1 = sqrt(-log(1 - r1) * 2.0) / sqrt(2.0);
  double ang1 = pi2 * r2;

  rn1 = slg1 * cos(ang1);
  rn2 = slg1 * sin(ang1);
}


//====================================================================
void RandomNumbers::uniform(Field& f)
{
  int Nin  = f.nin();
  int Nvol = f.nvol();
  int Nex  = f.nex();

  for (int ex = 0; ex < Nex; ++ex) {
    for (int site = 0; site < Nvol; ++site) {
      for (int in = 0; in < Nin; ++in) {
        double rn = get();
        f.set(in, site, ex, rn);
      }
    }
  }
}


//====================================================================
void RandomNumbers::gauss(Field& f)
{
  int Nin  = f.nin();
  int Nvol = f.nvol();
  int Nex  = f.nex();

  double rn1, rn2;

  if ((Nin % 2) == 0) {  // Nin is even
    int Nin2 = Nin / 2;
    for (int ex = 0; ex < Nex; ++ex) {
      for (int site = 0; site < Nvol; ++site) {
        for (int in = 0; in < Nin2; ++in) {
          //assert(2*in+1 < Nin);
          gauss(rn1, rn2);
          f.set(2 * in, site, ex, rn1);
          f.set(2 * in + 1, site, ex, rn2);
        }
      }
    }
  } else {
    for (int ex = 0; ex < Nex; ++ex) {
      for (int site = 0; site < Nvol; ++site) {
        for (int in = 0; in < Nin; ++in) {
          gauss(rn1, rn2);
          f.set(in, site, ex, rn1);
          // rn2 is not used
        }
      }
    }
  }
}


//====================================================================
void RandomNumbers::gauss_lex_global(Field& f)
{
  Index_lex idx;

  int Nin  = f.nin();
  int Nvol = f.nvol();
  int Nex  = f.nex();

  int Lvol = CommonParameters::Lvol();

  Field f_tmp(Nin, Lvol, Nex);

  gauss(f_tmp);

  int Lx  = CommonParameters::Lx();
  int Ly  = CommonParameters::Ly();
  int Lz  = CommonParameters::Lz();
  int Lt  = CommonParameters::Lt();
  int Nx  = CommonParameters::Nx();
  int Ny  = CommonParameters::Ny();
  int Nz  = CommonParameters::Nz();
  int Nt  = CommonParameters::Nt();
  int Ipx = Communicator::ipe(0);
  int Ipy = Communicator::ipe(1);
  int Ipz = Communicator::ipe(2);
  int Ipt = Communicator::ipe(3);

  for (int ex = 0; ex < Nex; ++ex) {
    for (int t = 0; t < Nt; ++t) {
      int t2 = t + Ipt * Nt;
      for (int z = 0; z < Nz; ++z) {
        int z2 = z + Ipz * Nz;
        for (int y = 0; y < Ny; ++y) {
          int y2 = y + Ipy * Ny;
          for (int x = 0; x < Nx; ++x) {
            int x2    = x + Ipx * Nx;
            int site  = idx.site(x, y, z, t);
            int site2 = x2 + Lx * (y2 + Ly * (z2 + Lz * t2));
            assert(site < Nvol);
            assert(site2 < Lvol);
            for (int in = 0; in < Nin; ++in) {
              f.set(in, site, ex, f_tmp.cmp(in, site2, ex));
            }
          }
        }
      }
    }
  }
}


//====================================================================
void RandomNumbers::uniform_lex_global(Field& f)
{
  Index_lex idx;

  int Nin  = f.nin();
  int Nvol = f.nvol();
  int Nex  = f.nex();

  int Lvol = CommonParameters::Lvol();

  Field f_tmp(Nin, Lvol, Nex);

  uniform(f_tmp);

  int Lx  = CommonParameters::Lx();
  int Ly  = CommonParameters::Ly();
  int Lz  = CommonParameters::Lz();
  int Lt  = CommonParameters::Lt();
  int Nx  = CommonParameters::Nx();
  int Ny  = CommonParameters::Ny();
  int Nz  = CommonParameters::Nz();
  int Nt  = CommonParameters::Nt();
  int Ipx = Communicator::ipe(0);
  int Ipy = Communicator::ipe(1);
  int Ipz = Communicator::ipe(2);
  int Ipt = Communicator::ipe(3);

  for (int ex = 0; ex < Nex; ++ex) {
    for (int t = 0; t < Nt; ++t) {
      int t2 = t + Ipt * Nt;
      for (int z = 0; z < Nz; ++z) {
        int z2 = z + Ipz * Nz;
        for (int y = 0; y < Ny; ++y) {
          int y2 = y + Ipy * Ny;
          for (int x = 0; x < Nx; ++x) {
            int x2    = x + Ipx * Nx;
            int site  = idx.site(x, y, z, t);
            int site2 = x2 + Lx * (y2 + Ly * (z2 + Lz * t2));
            assert(site < Nvol);
            assert(site2 < Lvol);
            for (int in = 0; in < Nin; ++in) {
              f.set(in, site, ex, f_tmp.cmp(in, site2, ex));
            }
          }
        }
      }
    }
  }
}


//====================================================================
void RandomNumbers::gauss_eo_global(Field& f)
{
  Index_eo idx;

  int Nin  = f.nin();
  int Nvol = f.nvol();
  int Nex  = f.nex();

  int Lvol2 = CommonParameters::Lvol() / 2;

  Field f_tmp(Nin, Lvol2, Nex);

  gauss(f_tmp);

  int Lx2 = CommonParameters::Lx() / 2;
  int Ly  = CommonParameters::Ly();
  int Lz  = CommonParameters::Lz();
  int Lt  = CommonParameters::Lt();
  int Nx2 = CommonParameters::Nx() / 2;
  int Ny  = CommonParameters::Ny();
  int Nz  = CommonParameters::Nz();
  int Nt  = CommonParameters::Nt();
  int Ipx = Communicator::ipe(0);
  int Ipy = Communicator::ipe(1);
  int Ipz = Communicator::ipe(2);
  int Ipt = Communicator::ipe(3);

  for (int ex = 0; ex < Nex; ++ex) {
    for (int t = 0; t < Nt; ++t) {
      int t2 = t + Ipt * Nt;
      for (int z = 0; z < Nz; ++z) {
        int z2 = z + Ipz * Nz;
        for (int y = 0; y < Ny; ++y) {
          int y2 = y + Ipy * Ny;
          for (int x = 0; x < Nx2; ++x) {
            int x2    = x + Ipx * Nx2;
            int site  = idx.site(x, y, z, t, 0);
            int site2 = x2 + Lx2 * (y2 + Ly * (z2 + Lz * t2));
            assert(site < Nvol);
            assert(site2 < Lvol2);
            for (int in = 0; in < Nin; ++in) {
              f.set(in, site, ex, f_tmp.cmp(in, site2, ex));
            }
          }
        }
      }
    }
  }
}


//====================================================================
//============================================================END=====
