/*!
        @file    test_Gauge_Plaquette.cpp

        @brief

        @author  Hideo Matsufuru  (matsufuru)
                 $LastChangedBy: aoyama $

        @date    $LastChangedDate: 2013-01-22 13:51:53 #$

        @version $LastChangedRevision: 2314 $
*/

#include "test.h"

#include "IO/gaugeConfig.h"

#include "Measurements/Gauge/staple.h"

#include "Tools/randomNumberManager.h"

//====================================================================
//! Test of gauge quantities.

/*!
                          [28 Dec 2011 H.Matsufuru]
    (Coding history will be recovered from trac.)
    YAML is implemented   [14 Nov 2012 Y.Namekawa]
    unique_ptr is introduced to avoid memory leaks
                          [21 Mar 2015 Y.Namekawa]
    Factory is introduced [24 Jan 2017 Y.Namekawa]
 */

namespace Test_Gauge {
  const std::string test_name = "Gauge.Plaquette";

  //- test-private parameters
  namespace {
    // const std::string filename_input = "test_Gauge.yaml";
  }

  //- prototype declaration
  // int plaquette(void);
  int plaquette(const std::string&);

  //- plaq for various types of index
  int plaquette_lex()
  {
    return plaquette("test_Gauge_lex.yaml");
  }


  int plaquette_eo()
  {
    return plaquette("test_Gauge_eo.yaml");
  }


#ifdef USE_TESTMANAGER_AUTOREGISTER
  namespace {
#if defined(USE_GROUP_SU2)
    // Nc=2 is not available.
#else
    static const bool is_registered_lex = TestManager::RegisterTest(
      "Gauge.Plaquette.lex",
      plaquette_lex
      );

    static const bool is_registered_eo = TestManager::RegisterTest(
      "Gauge.Plaquette.eo",
      plaquette_eo
      );
#endif
  }
#endif

  //====================================================================
  int plaquette(const std::string& filename_input)
  {
    // #####  parameter setup  #####
    const int Nvol = CommonParameters::Nvol();
    const int Ndim = CommonParameters::Ndim();

    const Parameters params_all = ParameterManager::read(filename_input);

    const Parameters params_test   = params_all.lookup("Test_Gauge");
    const Parameters params_staple = params_all.lookup("Staple");

    const string        str_gconf_status = params_test.get_string("gauge_config_status");
    const string        str_gconf_read   = params_test.get_string("gauge_config_type_input");
    const string        readfile         = params_test.get_string("config_filename_input");
    const string        str_rand_type    = params_test.get_string("random_number_type");
    const unsigned long seed             = params_test.get_unsigned_long("seed_for_random_number");
    const string        str_vlevel       = params_test.get_string("verbose_level");

    const bool   do_check        = params_test.is_set("expected_result");
    const double expected_result = do_check ? params_test.get_double("expected_result") : 0.0;

    const string str_index_type = params_staple.get_string("index_type");

    const Bridge::VerboseLevel vl = vout.set_verbose_level(str_vlevel);

    //- print input parameters
    vout.general(vl, "  gconf_status = %s\n", str_gconf_status.c_str());
    vout.general(vl, "  gconf_read   = %s\n", str_gconf_read.c_str());
    vout.general(vl, "  readfile     = %s\n", readfile.c_str());
    vout.general(vl, "  rand_type    = %s\n", str_rand_type.c_str());
    vout.general(vl, "  seed         = %lu\n", seed);
    vout.general(vl, "  vlevel       = %s\n", str_vlevel.c_str());
    vout.general(vl, "\n");

    //- input parameter check
    int err = 0;
    err += ParameterCheck::non_NULL(str_gconf_status);

    if (err) {
      vout.crucial(vl, "Error at %s: input parameters have not been set\n", test_name.c_str());
      exit(EXIT_FAILURE);
    }

    RandomNumberManager::initialize(str_rand_type, seed);


    // ####  Set up a gauge configuration  ####
    Field_G U(Nvol, Ndim);

    if (str_gconf_status == "Continue") {
      GaugeConfig(str_gconf_read).read(U, readfile);
    } else if (str_gconf_status == "Cold_start") {
      GaugeConfig("Unit").read(U);
    } else if (str_gconf_status == "Hot_start") {
      GaugeConfig("Random").read(U);
    } else {
      vout.crucial(vl, "Error at %s: unsupported gconf status \"%s\"\n", test_name.c_str(), str_gconf_status.c_str());
      exit(EXIT_FAILURE);
    }


    // #### object setup #####
    unique_ptr<Staple> staple(Staple::New(str_index_type));
    staple->set_parameters(params_staple);

    Timer timer(test_name);


    // ####  Execution main part  ####
    timer.start();

    const double result = staple->plaquette(U);

    timer.report();

    RandomNumberManager::finalize();


    if (do_check) {
      return Test::verify(result, expected_result);
    } else {
      vout.detailed(vl, "check skipped: expected_result not set.\n\n");
      return EXIT_SKIP;
    }
  }
} // namespace Test_Gauge
